package com.xebialabs.deployit.filter

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.engine.spi.exception.DeployitException

import javax.servlet.FilterChain
import javax.servlet.http.{Cookie, HttpServletRequest, HttpServletResponse}
import org.springframework.security.web.csrf.CsrfToken
import org.springframework.web.filter.OncePerRequestFilter
import org.springframework.web.util.WebUtils

class CsrfHeaderFilter extends OncePerRequestFilter {
  val DEFAULT_CSRF_COOKIE_NAME = "XSRF-TOKEN"

  val DEFAULT_CSRF_HEADER_NAME = "X-XSRF-TOKEN"

  val cookieName: String = DEFAULT_CSRF_COOKIE_NAME

  val serverConfiguration: ServerConfiguration = ServerConfiguration.getInstance

  override protected def doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain): Unit = {
    val csrf = request.getAttribute(classOf[CsrfToken].getName) match {
      case c: CsrfToken if c != null => c
      case other => throw new DeployitException(s"Expected CsrfToken non null - but extracted ${other.getClass.getSimpleName} - $other")
    }
    val cookie = WebUtils.getCookie(request, cookieName)
    val token = csrf.getToken
    if (cookie == null || token != null && !(token == cookie.getValue)) {
        val newCookie = new Cookie(cookieName, token)
        newCookie.setPath(getRequestContext(request))
        if (serverConfiguration.isSsl) {
          newCookie.setSecure(true)
        }
        response.addCookie(newCookie)
    }
    filterChain.doFilter(request, response)
  }

  private def getRequestContext(request: HttpServletRequest): String = {
    val contextPath = request.getContextPath
    if (contextPath.nonEmpty) {
      contextPath
    }
    else {
      "/"
    }
  }
}
