package com.xebialabs.deployit.service.dependency

import com.github.zafarkhaja.semver.{Version => OsgiVersion}
import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.{ProvisioningPackage, Application => UdmApplication, DeploymentPackage => UdmPackage, Version => UdmVersion}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.DependencyConstraints.DependencyException
import com.xebialabs.deployit.service.dependency.Implicits._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions
import scala.util.Try

@Component
class DependencyFinder @Autowired()(val repositoryService: RepositoryService) extends RepositoryServiceAware with Logging {

  case class Dependency(range: String, udmApplication: Option[UdmApplication], udmPackage: Option[UdmPackage])

  type VersionPicker = (String, List[UdmPackage], VersionRange) => Option[UdmPackage]

  private[this] val NONE_VERSIONPICKER: VersionPicker = (_, _, _) => None

  def find(deploymentPackage: UdmPackage, versionPicker: VersionPicker = NONE_VERSIONPICKER): ApplicationGraph = {
    implicit val allDependencies: ApplicationGraph = graph()
    implicit val addedNodes: mutable.Buffer[String] = mutable.Buffer[String]()
    findDependencies(
      Application.main(deploymentPackage.getApplication.getId, deploymentPackage.getVersion),
      deploymentPackage.getApplicationDependencies.asScala.toMap,
      deploymentPackage.isInstanceOf[ProvisioningPackage],
      versionPicker
    )
    allDependencies
  }

  def findAndNormalize(deploymentPackage: UdmPackage, versionPicker: VersionPicker = NONE_VERSIONPICKER): List[List[UdmPackage]] = {
    val dependencies = find(deploymentPackage, versionPicker)
    DependencyConstraints.validateApplicationGraph(dependencies)
    dependencies.allRequiredVersionsInTopologicalOrder
  }

  private[this] def findDependencies(app: Application, dependencies: Map[String, String], isMainPackageProvisioning: Boolean, versionPicker: VersionPicker)
                                    (implicit applicationDependencies: ApplicationGraph, addedNodes: mutable.Buffer[String]): Unit = {
    if (!addedNodes.contains(app.id)) {
      addedNodes += app.id
      val resolved = resolveDependencies(dependencies, app, isMainPackageProvisioning, versionPicker)
      applicationDependencies.add(app)
      findTransientDependencies(resolved, isMainPackageProvisioning, versionPicker)
      findDirectDependencies(app, resolved)
    }
  }

  private[this] def findDirectDependencies(app: Application, resolvedDependencies: Map[String, Dependency])
                                          (implicit applicationDependencies: ApplicationGraph): Unit = {
    import scalax.collection.edge.Implicits._
    resolvedDependencies.foreach { case (appName, Dependency(range, udmApp, udmPackage)) =>
      applicationDependencies.add((app ~+> node(appName, udmApp)) (Label(udmPackage, range)))
    }
  }

  private[this] def findTransientDependencies(resolvedDependencies: Map[String, Dependency], isMainPackageProvisioning: Boolean, versionPicker: VersionPicker)
                                             (implicit applicationDependencies: ApplicationGraph, addedNodes: mutable.Buffer[String]): Unit = {
    resolvedDependencies.collect {
      case (_, Dependency(_, Some(udmApp), Some(udmPackage))) =>
        findDependencies(Application.resolved(udmApp.getId), udmPackage.getApplicationDependencies.asScala.toMap, isMainPackageProvisioning, versionPicker)
    }
  }

  private[this] def resolveDependencies(dependencies: Map[String, String], dependingApplication: Application, isMainPackageProvisioning: Boolean, versionPicker: VersionPicker): Map[String, Dependency] = {
    def findByRange(a: UdmApplication, packages: List[UdmPackage], range: String): Option[UdmPackage] = {
      val versionRange: VersionRange = new VersionRange(range)
      versionPicker(a.getId, packages, versionRange).orElse(findMaxVersion(packages, versionRange))
    }

    def findByFixed(packages: List[UdmPackage], version: String): Option[UdmPackage] = packages.find(_.getName == version)

    dependencies.map {
      case (appName, dep) =>
        val app = loadApplication(appName, dependingApplication)
        val version: Option[UdmPackage] = app.flatMap { a =>
          val versions = repositoryService.listEntities[UdmVersion](typedSearchParameters[UdmVersion].setParent(a.getId), 2, true).asScala.toList
          val packages = versions.collect { case pck: UdmPackage if pck.isInstanceOf[ProvisioningPackage] == isMainPackageProvisioning => pck }
          val maybePackage = Try(findByRange(a, packages, dep)).getOrElse(findByFixed(packages, dep))
          if (maybePackage.isEmpty && versions.size > packages.size) {
            val msg = if (isMainPackageProvisioning) "Deployment Packages" else "Provisioning Packages"
            throw new DependencyException(s"""Error while trying to resolve the dependencies of application "$dependingApplication". $msg were found for application "$appName" in the given range ($dep). Dependencies across different package types is not supported.""".stripMargin)
          }
          maybePackage
        }
        appName -> Dependency(dep, app, version)
    }
  }

  private[this] def loadApplication(name: String, dependingApplication: Application): Option[UdmApplication] = {
    val applications = repositoryService.listEntities[UdmApplication](typedSearchParameters[UdmApplication].setName(name), 2, true)
    if (applications.size() > 1)
      throw new DependencyException(s"""Error while trying to resolve the dependencies of application "$dependingApplication". Multiple applications found with the name "$name". Please rename one of your applications, application names should be unique.""")
    applications.asScala.headOption
  }

  private[this] def findMaxVersion(versions: List[UdmPackage], range: VersionRange): Option[UdmPackage] = {
    val resolvedVersions = versions.map(uv => uv -> Try(uv.toOsgi).toOption).toMap
    val maxOsgi = resolvedVersions.values.flatten.foldLeft(Option.empty[OsgiVersion])((result, v) => result max v.in(range))
    resolvedVersions.collectFirst { case (deploymentPackage, `maxOsgi`) => deploymentPackage }
  }

  private[this] def node(appName: String, app: Option[UdmApplication]) = app.map(a => Application.resolved(a.getId)).getOrElse(Application.unresolved(appName))
}
