package com.xebialabs.deployit.service.dependency

import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.DeployedApplicationsFinder.DeployedApplications
import com.xebialabs.deployit.service.dependency.Implicits._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import scalax.collection.GraphPredef._

object DeployedApplicationsFinder {

  class DeployedApplications(_deployedGraph: DeployedApplicationGraph) {

    def findDependentApplications(deployedApplication: DeployedApplication): Seq[DeployedApplicationInfo] = {
      val b = findDeployedApplicationInfo(deployedApplication.getId).map(graphNode).map { node =>
        node.incoming.map(edge => edge.head.value).toSeq
      }
      b.getOrElse(Nil)
    }

    def findRemovableDeployedApplicationsGroups(deployedApplication: DeployedApplication): List[List[DeployedApplicationInfo]] = {
      val maybeDeployedInfo = findDeployedApplicationInfo(deployedApplication.getId).map(graphNode)
      val graph = maybeDeployedInfo.map { mainNode =>
        val mainAppAndDepsGraph = _deployedGraph.filter(_deployedGraph.having(node = n => n.hasPredecessor(mainNode) || n == mainNode))
        val independentApps = _deployedGraph.nodes.toOuter.diff(mainAppAndDepsGraph.nodes.toOuter)
        independentApps.foldLeft(_deployedGraph)(excludeNodesAndDepsFromGraph)
      }
      graph.map(_.dependencyApplicationGroups).getOrElse(Nil)
    }

    private def graphNode(deployedApplicationInfo: DeployedApplicationInfo) = _deployedGraph.get(deployedApplicationInfo)

    private def excludeNodesAndDepsFromGraph(graph: DeployedApplicationGraph, deployedApplicationInfo: DeployedApplicationInfo) = {
      if (graph.contains(deployedApplicationInfo)) {
        val node = graph.get(deployedApplicationInfo)
        graph.filter(graph.having(node = n => !n.hasPredecessor(node) && n != node))
      } else {
        graph
      }
    }

    private def findDeployedApplicationInfo(deployedApplicationId: String): Option[DeployedApplicationInfo] = {
      val nodes: Iterable[DeployedApplicationInfo] = _deployedGraph.nodes.map(_.value)
      nodes.find(_.deployedApplicationId == deployedApplicationId)
    }

    def findApplicationInfo(applicationId: String): Option[DeployedApplicationInfo] = {
      val nodes: Iterable[DeployedApplicationInfo] = _deployedGraph.nodes.map(_.value)
      nodes.find(_.applicationId == applicationId)
    }
  }

}

@Component
class DeployedApplicationsFinder @Autowired()(val repositoryService: RepositoryService) extends DeployedApplicationInfoLoader with Logging {

  def allDeployedApplicationsWithDependencies(environmentId: String): DeployedApplications = {
    implicit val _deployedGraph = deployedGraph()
    val allDeployedApplications: List[DeployedApplicationInfo] = loadDeployedApplications(environmentId)
    implicit val infoPerAppName: Map[String, DeployedApplicationInfo] = allDeployedApplications.map(inf => inf.applicationId.name -> inf).toMap
    infoPerAppName.values.foreach(addDeployedApplicationToGraph)
    new DeployedApplications(_deployedGraph)
  }

  def addDeployedApplicationToGraph(deployedInfo: DeployedApplicationInfo)(implicit _deployedGraph: DeployedApplicationGraph, infoPerAppName: Map[String, DeployedApplicationInfo]): Unit = {
    _deployedGraph.add(deployedInfo)
    deployedInfo.applicationDependencies.foreach { case (dependencyAppId, _) =>
      addEdgeToGraphIfPresent(deployedInfo, infoPerAppName.get(dependencyAppId.name))
    }
  }

  def addEdgeToGraphIfPresent(deployedInfo: DeployedApplicationInfo, maybeDependencyInfo: Option[DeployedApplicationInfo])(implicit _deployedGraph: DeployedApplicationGraph): Unit = {
    maybeDependencyInfo.foreach { dependencyInfo =>
      _deployedGraph.add(deployedInfo ~> dependencyInfo)
    }
  }

}
