package com.xebialabs.deployit.service.controltask

import ai.digital.deploy.task.status.queue.TaskPathStatusListener

import java.util
import ai.digital.deploy.tasker.common.TaskMetadata._
import ai.digital.deploy.tasker.common.TaskType
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.engine.api.dto.Control
import com.xebialabs.deployit.engine.spi.event.{TaskCreatedEvent, TaskStartedEvent}
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, MethodDescriptor}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItemWithPolicies
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, OnTaskFailurePolicy, OnTaskSuccessPolicy, Parameters}
import com.xebialabs.deployit.repository.{RepositoryService, WorkDir, WorkDirFactory}
import com.xebialabs.deployit.service.externalproperties.ExternalPropertiesResolver
import com.xebialabs.deployit.spring.BeanWrapper
import com.xebialabs.deployit.task.WorkdirCleanerTrigger
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import scala.collection.mutable.ArrayBuffer
import scala.jdk.CollectionConverters._

@Component
class ControlTaskServiceImpl(@Autowired val repositoryService: RepositoryService,
                             @Autowired val externalPropertiesResolver: ExternalPropertiesResolver,
                             @Autowired val engine: BeanWrapper[TaskExecutionEngine],
                             @Autowired val workDirFactory: WorkDirFactory) extends ControlTaskService {
  override def prepare(controlName: String, id: String): Control = {
    val configurationItem: ConfigurationItem = repositoryService.read(id)
    externalPropertiesResolver.resolveExternalProperties(configurationItem)
    val controlTask = Option(configurationItem.getType.getDescriptor.getControlTask(controlName))
      .getOrElse(throw InvalidControlException(
        "Control [%s] could  not be found on type [%s]", controlName, configurationItem.getType))
    val params = if (controlTask.getParameterObjectType != null)
      Option(controlTask.getParameterObjectType.getDescriptor.newInstance("parameters"))
    else
      None
    new Control(configurationItem, controlName, params.orNull)
  }

  override def createTaskSpec(workDir: WorkDir,
                              ci: ConfigurationItem,
                              controlTaskName: TaskId,
                              params: Parameters,
                              owner: Authentication): TaskSpecification = {
    val method: MethodDescriptor = DescriptorRegistry.getDescriptor(ci.getType).getControlTask(controlTaskName)
    checkArgument(method != null, "ConfigurationItem %s of type %s does not have a control task named %s.", ci.getId, ci.getType, controlTaskName)

    val steps: util.List[Step] = method.invoke(ci, params)

    val (onSuccessPolicy, onFailurePolicy) = ci match {
      case ciWithPolicy: BaseConfigurationItemWithPolicies => (ciWithPolicy.getOnSuccessPolicy, ciWithPolicy.getOnFailurePolicy)
      case _ => (OnTaskSuccessPolicy.NOOP, OnTaskFailurePolicy.NOOP)
    }

    val desc: String = descriptionFor(ci, controlTaskName)
    val plan: PhasedPlan = asPhasedPlan(controlTaskName, buildExecutionPlan(ci, controlTaskName, steps.asScala.toList))
    val planForSatellites: PhasedPlan = Satellites.prepareForSatelliteExecution(plan)
    val phaseContainer = Plans.toBlockBuilder(planForSatellites).build()

    val taskSpec = new TaskSpecification(
      desc,
      owner,
      workDir,
      phaseContainer,
      onSuccessPolicy,
      onFailurePolicy
    )
    taskSpec.getListeners.add(new WorkdirCleanerTrigger(workDir))
    taskSpec.getListeners.add(new TaskPathStatusListener());

    val currentCi: ConfigurationItem = repositoryService.read(ci.getId)
    taskSpec.getMetadata.putAll(Map(
      TASK_TYPE -> TaskType.CONTROL.name,
      TASK_NAME -> controlTaskName,
      TASK_LABEL -> method.getLabel,
      CONTROL_TASK_TARGET_CI -> ci.getId,
      CONTROL_TASK_TARGET_INTERNAL_CI -> currentCi.get$internalId().toString,
      CONTROL_TASK_TARGET_SECURED_CI -> Option(currentCi.get$securedCi()).map(_.toString).orNull,
      CONTROL_TASK_TARGET_DIRECTORY_REFERENCE -> currentCi.get$directoryReference()
    ).asJava)

    taskSpec
  }

  override def create(control: Control, owner: Authentication, workDir: WorkDir): String = {
    val ci = control.getConfigurationItem
    val controlTaskName = control.getControlName
    val params = control.getParameters
    val taskSpec = createTaskSpec(workDir, ci, controlTaskName, params, owner)
    val taskId = engine.get().register(taskSpec)
    EventBusHolder.publish(new TaskCreatedEvent(taskId, controlTaskName, ci.getId))
    taskId
  }

  override def start(taskId: String): Unit = {
    engine.get().execute(taskId)
    EventBusHolder.publish(new TaskStartedEvent(taskId))
  }

  private[controltask] def buildExecutionPlan(ci: ConfigurationItem, controlTaskName: String, steps: List[Step]) = {
    steps match {
      case Nil => throw new IllegalStateException(s"Control task $controlTaskName on ${ci.getId} resulted in no steps.")
      case head :: _ =>
        val aggregator = (satelliteOfStep(head), ArrayBuffer(head)) :: Nil
        val folded = steps.tail.foldLeft(aggregator)({
          case (acc, step) if acc.head._1 == satelliteOfStep(step) =>
            acc.head._2.append(step)
            acc
          case (acc, step) => (satelliteOfStep(step), ArrayBuffer(step)) :: acc
        })
        val builders = folded.map({ case (sat, ss) => new StepPlan(descriptionFor(ci, controlTaskName, sat), ss.toList.asJava, List[ExecutionStateListener]().asJava, sat.orNull) }).reverse
        builders match {
          case hd :: Nil => hd
          case xs => new SerialPlan(descriptionFor(ci, controlTaskName), xs.asJava, null)
        }
    }
  }

  private[this] def satelliteOfStep(step: Step): Option[Satellite] = Option(step).collect({ case sa: SatelliteAware if sa.getSatellite != null => sa.getSatellite })

  private[this] def asPhasedPlan(controlTaskName: String, plan: ExecutablePlan): PhasedPlan = {
    val phase: PlanPhase = new PlanPhase(plan, s"Execute the $controlTaskName control task", List().asJava, false)
    new PhasedPlan(List(phase).asJava, List().asJava)
  }

  private[this] def descriptionFor(ci: ConfigurationItem, controlTaskName: String, satellite: Option[Satellite]): String = {
    satellite match {
      case Some(x) => descriptionFor(ci, controlTaskName) + s" (executed on $x)"
      case None => descriptionFor(ci, controlTaskName)
    }
  }

  private[this] def descriptionFor(ci: ConfigurationItem, controlTaskName: String) = s"Control task [$controlTaskName] for ${ci.getId}"
}

@HttpResponseCodeResult(statusCode = 400)
case class InvalidControlException(messageTemplate: String, params: Any*) extends DeployitException(messageTemplate, params)
