package com.xebialabs.deployit.service.deployment;

import ai.digital.deploy.core.common.TypeCalculator;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.service.replacement.Dictionaries;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

import static java.util.stream.Collectors.groupingBy;

@Component
public class DeployedProcessorsFactory {

    private final DeploymentContext.BeanSupplier beanSupplier;
    private final TypeCalculator calculator = new TypeCalculator();
    private final Map<Type, Class<? extends DeployedGenerator>> deployableSpecificGenerators = new HashMap<>();

    @Autowired
    public DeployedProcessorsFactory(DeployedArtifactPlaceholdersResolver deployedArtifactPlaceholdersResolver,
                                     DeployedPropertySetter deployedPropertySetter) {
        final Map<Class<?>, Object> beans = new HashMap<>();
        beans.put(deployedArtifactPlaceholdersResolver.getClass(), deployedArtifactPlaceholdersResolver);
        beans.put(deployedPropertySetter.getClass(), deployedPropertySetter);
        beans.put(calculator.getClass(), calculator);
        beanSupplier = new DeploymentContext.BeanSupplier() {
            @Override
            public <T> T getBean(Class<T> beanClass) {
                return beanClass.cast(beans.get(beanClass));
            }
        };
    }

    public DeployedGenerator createTagDeployedGenerator() {
        return new TagMatchingDeployedGenerator(createCoreDeployedGenerator());
    }

    public DeployedGenerator createCoreDeployedGenerator() {
        return new ProvisioningPlaceholderPropertyResolverDeployedGenerator(
                perTypeDeployedGenerator(
                        new PlaceholderPropertyDeployedGenerator(
                                new PropertyResolverDeployedGenerator(
                                        new TypeFilterDeployedGenerator(
                                                new CardinalDeployedGenerator(new BasicDeployedGenerator())
                                        )
                                )
                        )
                )
        );
    }

    public DeployedGenerator createUpgradeDeployedGenerator() {
        return new ProvisioningPlaceholderPropertyResolverDeployedGenerator(
                perTypeDeployedGenerator(
                        new UpdateSameDeployableChainGenerator(
                                new PlaceholderPropertyDeployedGenerator(
                                        new PropertyResolverDeployedGenerator(
                                                new CardinalDeployedGenerator(new BasicDeployedGenerator())
                                        )
                                )
                        )
                )
        );
    }

    private PerDeployableTypeGenerator perTypeDeployedGenerator(DeployedGenerator generator) {
        return new PerDeployableTypeGenerator(deployableSpecificGenerators, generator);
    }

    public DeploymentContext createContextForSpecificType(Deployment deployment, Dictionaries dictionaries, Type deployedType) {
        return new DeploymentContext((DeployedApplication) deployment.getDeployedApplication(), dictionaries, beanSupplier,
                (deployable, container) -> deployedType, deployable -> Collections.emptyList());
    }

    public DeploymentContext createContextWithCalculatedType(Deployment deployment, Dictionaries dictionaries) {
        return new DeploymentContext((DeployedApplication) deployment.getDeployedApplication(), dictionaries, beanSupplier,
                (deployable, container) -> calculator.findMostSpecificDeployedType(deployable.getType(), container.getType()),
                deployable -> Collections.emptyList());
    }

    public DeploymentContext createContextForUpgrade(DeployedApplication deployedApplication, Dictionaries dictionaries, Set<Deployed> previousDeployeds) {
        Map<String, List<Deployed>> deployedsPerDeployableId = previousDeployeds.stream().filter(x -> x.getDeployable() != null).collect(groupingBy(deployed -> deployed.getDeployable().getName()));
        return new DeploymentContext(deployedApplication, dictionaries, beanSupplier,
                (deployable, container) -> calculator.findMostSpecificDeployedType(deployable.getType(), container.getType()),
                deployable -> deployedsPerDeployableId.get(deployable.getName())
        );
    }
}
