package com.xebialabs.deployit.service.deployment.setter;

import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;

/**
 * This property setter is a hidden functionality, which is used only by LFG.
 * It is only used when the hidden property <i>server.mapping.override.deployed.fields.on.update.previous.behaviour</i> is added into 'xl-deploy.conf'.
 * See more information here <a href="https://xebialabs.atlassian.net/browse/DEPL-12902">DEPL-12902</a>.
 */
@Component
public class PreviousBehaviourPropertySetter extends AbstractPropertySetter {
    public static final String PREVIOUS_BEHAVIOUR_KEY = "server.mapping.override.deployed.fields.on.update.previous.behaviour";

    private static final Logger logger = LoggerFactory.getLogger(PreviousBehaviourPropertySetter.class);

    @Autowired
    public PreviousBehaviourPropertySetter(RepositoryService repository) {
        super(repository);
    }

    private void checkWhetherValueIsSecureAndPropertyIsPassword(Object value, PropertyDescriptor deployedDesc, Object deployablePropertyValue) throws DictionaryValueException {
        if (value instanceof EncryptedStringValue && !deployedDesc.isPassword()) {
            logger.warn("The deployable value [{}] resolved to an encrypted value, but property [{}] is not a password field.", deployablePropertyValue, deployedDesc.getFqn());
            throw new DictionaryValueException("The deployable value [%s] resolved to an encrypted value, but property [%s] is not a password field.", deployablePropertyValue, deployedDesc.getFqn());
        }
    }

    @Override
    public void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                            ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, PropertyDescriptor sourcePropertyDescriptor) {
        String targetPropertyName = targetPropertyDescriptor.getName();
        String sourcePropertyName = sourcePropertyDescriptor != null ? sourcePropertyDescriptor.getName() : null;
        Object previousTargetValue = getValueIfExists(previousTarget, targetPropertyName);
        Object previousSourceValue = getValueIfExists(previousSource, sourcePropertyName);
        Object currentSourceValue = getValueIfExists(currentSource, sourcePropertyName);

        Object valueToSet = previousTargetValue;
        PropertyDescriptor valueToSetPropDesc = targetPropertyDescriptor;

        if (previousTargetValue != null) {
            setSilently(currentTarget, targetPropertyDescriptor, previousTargetValue);
            return;
        }

        // If the old deployable value had placeholder, always use the newly set one.
        final boolean oldDeployableHadPLaceholder = previousSourceValue != null && MustachePlaceholderScanner.hasPlaceholders(previousSourceValue, sourcePropertyDescriptor);

        // If the old deployable value was not set, and now it is, use the newly set one.
        final boolean oldDeployableValueNotSet = previousSourceValue == null && currentSourceValue != null;

        if (oldDeployableHadPLaceholder || oldDeployableValueNotSet) {
            valueToSet = currentSourceValue;
            valueToSetPropDesc = sourcePropertyDescriptor;
        }

        if (isNullOrEmpty(valueToSet, valueToSetPropDesc)) {
            setDeployedFromDictionary(currentTarget, dictionary, targetPropertyDescriptor, sourcePropertyDescriptor);
            return;
        }

        try {
            Set<String> missingPlaceholdersAggregator = new HashSet<>();
            Object deployedPropertyValue = dictionary.resolve(valueToSet, valueToSetPropDesc, missingPlaceholdersAggregator);
            checkWhetherValueIsSecureAndPropertyIsPassword(deployedPropertyValue, targetPropertyDescriptor, valueToSet);
            setSilently(currentTarget, targetPropertyDescriptor, convertIfNeeded(deployedPropertyValue, valueToSetPropDesc, targetPropertyDescriptor));
        } catch (DictionaryValueException e) {
            logger.info("Could not resolve dictionary keys for property {} on {}.", sourcePropertyDescriptor, currentTarget);
            logger.trace("Exception was", e);
        }
    }
}
