package com.xebialabs.deployit.service.deployment;

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Taggable;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Set;

import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.google.common.collect.Sets.newHashSet;

public class DeploymentOperationCalculator {
    private DeploymentOperationCalculator() {}

    @SuppressWarnings("rawtypes")
    public static DeltaSpecificationBuilder calculate(DeltaSpecificationBuilder builder, Set<? extends Deployed> oldDeployeds, Set<? extends Deployed> newDeployeds) {
        checkNotNull(oldDeployeds, "Cannot calculate on null old Deployeds");
        checkNotNull(newDeployeds, "Cannot calculate on null new Deployeds");

        if (oldDeployeds.isEmpty()) {
            for (Deployed newDeployed : newDeployeds) {
                builder.create(newDeployed);
            }
        } else if (newDeployeds.isEmpty()) {
            for (Deployed oldDeployed : oldDeployeds) {
                builder.destroy(oldDeployed);
            }
        } else {
            calculateUpgrade(builder, oldDeployeds, newDeployeds);
        }

        return builder;
    }

    public static DeltaSpecificationBuilder deltaToRedeploy(DeltaSpecificationBuilder builder, Set<? extends Deployed> oldDeployeds, Set<? extends Deployed> newDeployeds) {
        checkNotNull(oldDeployeds, "Cannot calculate on null old Deployeds");
        checkNotNull(newDeployeds, "Cannot calculate on null new Deployeds");

        oldDeployeds.stream().forEach(oldDeployed -> builder.destroy(oldDeployed));
        newDeployeds.stream().forEach(newDeployed -> builder.create(newDeployed));
        return builder;
    }

    @SuppressWarnings("rawtypes")
    static void calculateUpgrade(DeltaSpecificationBuilder builder, Set<? extends Deployed> oldDeployeds, Set<? extends Deployed> newDeployeds) {
        Set<? extends Deployed> olds = newHashSet(oldDeployeds);
        Set<? extends Deployed> news = newHashSet(newDeployeds);

        for (Deployed old : olds) {
            Deployed aNew = findSimilar(old, news);
            if (aNew != null) {
                news.remove(aNew);
                if (isDifferent(old, aNew)) {
                    builder.modify(old, aNew);
                } else {
                    builder.noOp(old, aNew);
                }
            } else {
                builder.destroy(old);
            }
        }

        for (Deployed aNew : news) {
            builder.create(aNew);
        }
    }

    @SuppressWarnings("rawtypes")
    static Deployed findSimilar(Deployed old, Set<? extends Deployed> news) {
        for (Deployed aNew : news) {
            if (isSimilar(old, aNew)) {
                return aNew;
            }
        }
        return null;
    }

    static boolean isDifferent(Deployed old, Deployed aNew) {
        logger.debug("Comparing {} to {}", old, aNew);
        if (!old.getType().equals(aNew.getType())) {
            logger.debug("Difference detected: Types do not match: {} <-> {}", old.getType(), aNew.getType());
            return true;
        }

        Descriptor d = DescriptorRegistry.getDescriptor(old.getType());
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            if (pd.getName().equals(Deployed.DEPLOYABLE_FIELD) ||
                pd.getName().equals(Deployed.CONTAINER_FIELD) ||
                pd.isHidden() ||
                pd.isTransient()) {
                logger.debug("Skipping property {} in comparison of {}", pd.getFqn(), aNew.getId());
                continue;
            }
            if (!pd.areEqual(old, aNew)) {
                logger.debug("Difference detected: Property {} does not match", pd.getFqn());
                logger.trace("- old Property value {}", pd.get(old));
                logger.trace("- new Property value {}", pd.get(aNew));
                return true;
            }
        }

        return isDifferent(old.getDeployable(), aNew.getDeployable());
    }

    private static boolean isSourceArtifact(Descriptor d) {
        return d.getType().isSubTypeOf(Type.valueOf(SourceArtifact.class));
    }

    static boolean isDifferent(Deployable oldDeployable, Deployable newDeployable) {
        if (!oldDeployable.getType().equals(newDeployable.getType())) {
            return true;
        }

        Descriptor d = DescriptorRegistry.getDescriptor(oldDeployable.getType());
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            if (pd.getName().equals(Taggable.TAGS_FIELD)
                    || (isSourceArtifact(d) && pd.getName().equals(SourceArtifact.FILE_URI_PROPERTY_NAME))
                    || pd.isHidden()) {
                continue;
            }
            if (!pd.areEqual(oldDeployable, newDeployable, ConfigurationItem::getName)
                    || (isSourceArtifact(d) && pd.getName().equals(SourceArtifact.IS_RESCANNED_PROPERTY_NAME) && pd.get(newDeployable).equals(true))) {
                return true;
            }
        }

        return false;
    }

    @SuppressWarnings("rawtypes")
    static boolean isSimilar(Deployed old, Deployed aNew) {
        return aNew.getId().equals(old.getId()) && old.getType().equals(aNew.getType());
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentOperationCalculator.class);
}
