package com.xebialabs.deployit.service.deployment.setter;

import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import static com.xebialabs.deployit.plugin.api.validation.ValidationMessage.warn;
import static java.util.stream.Collectors.joining;

@Component
public class DefaultPropertySetter extends AbstractPropertySetter {
    private static final Logger logger = LoggerFactory.getLogger(DefaultPropertySetter.class);

    @Autowired
    public DefaultPropertySetter(RepositoryService repository) {
        super(repository);
    }

    private boolean isPropertyValueEmptyOrNull(ConfigurationItem ci, String propertyName) {
        if (ci == null || !ci.hasProperty(propertyName)) {
            return true;
        }
        Object value = ci.getProperty(propertyName);
        PropertyDescriptor propertyDescriptor = ci.getType().getDescriptor().getPropertyDescriptor(propertyName);
        return isNullOrEmpty(value, propertyDescriptor);
    }

    private void setValidationMessages(ConfigurationItem target, PropertyDescriptor targetPropertyDescriptor, String propertyName, String message) {
        logger.error("Could not resolve dictionary keys for property " + targetPropertyDescriptor + " on " + target + ".");
        target.get$validationMessages().add(warn(target.getId(), propertyName, String.format("Could not resolve the property value(s) %s using the dictionary", message)));
    }

    private void resolveSourceValueFromDictionary(ConfigurationItem target, Object sourceValue, ConsolidatedDictionary dictionary, PropertyDescriptor sourcePropertyDescriptor, PropertyDescriptor targetPropertyDescriptor) {
        String propertyName = targetPropertyDescriptor.getName();
        try {
            Set<String> missingPlaceholdersAggregator = new HashSet<>();
            Object targetValue = dictionary.resolve(sourceValue, sourcePropertyDescriptor, missingPlaceholdersAggregator);
            if (targetValue instanceof EncryptedStringValue && !targetPropertyDescriptor.isPassword()) {
                logger.warn("The source value [{}] resolved to an encrypted value, but property [{}] is not a password field.", sourceValue, targetPropertyDescriptor.getFqn());
                ValidationMessage validationMessage = warn(target.getId(), propertyName, "The deployable value resolved to an encrypted value, but this field is not a password");
                target.get$validationMessages().add(validationMessage);
                return;
            }
            setSilently(target, targetPropertyDescriptor, convertIfNeeded(targetValue, sourcePropertyDescriptor, targetPropertyDescriptor));
            if(!missingPlaceholdersAggregator.isEmpty()){
                setValidationMessages(target, targetPropertyDescriptor, propertyName, missingPlaceholdersAggregator.stream().collect(joining("}}\", \"{{", "\"{{", "}}\"")));
            }
        } catch (DictionaryValueException e) {
            setValidationMessages(target, targetPropertyDescriptor, propertyName, Optional.ofNullable(sourceValue).map(s -> String.format("\"%s\"", s)).orElse(null));
        }
    }

    private void resolveFromSource(ConfigurationItem source, ConfigurationItem target, PropertyDescriptor sourcePropertyDescriptor, PropertyDescriptor targetPropertyDescriptor, ConsolidatedDictionary dictionary) {

        // Resolve from source
        logger.trace("Resolving property {} from source {} with type {}", targetPropertyDescriptor.getFqn(), source.getId(), source.getType());
        Object deployableValue = sourcePropertyDescriptor.get(source);
        // If source has no value, try dictionary propertyname fallback
        if (isNullOrEmpty(deployableValue, sourcePropertyDescriptor)) {
            logger.trace("Checking whether {} occurs in dictionary", targetPropertyDescriptor.getFqn());
            setDeployedFromDictionary(target, dictionary, targetPropertyDescriptor, sourcePropertyDescriptor);
            return;
        }
        // resolve source value through dictionary
        resolveSourceValueFromDictionary(target, deployableValue, dictionary, sourcePropertyDescriptor, targetPropertyDescriptor);
    }

    private void resolveFromDictionary(ConfigurationItem currentDeployed, ConsolidatedDictionary dictionary, PropertyDescriptor deployedPropDesc, PropertyDescriptor deployablePropDesc) {
        Object value = dictionary.get(deployedPropDesc.getFqn());
        value = convertIfNeeded(value, deployablePropDesc, deployedPropDesc);
        deployedPropDesc.set(currentDeployed, value);
    }

    private boolean isPropertyOnPreviousAndCurrentDeployableEmptyOrNull(ConfigurationItem currentDeployable, ConfigurationItem previousDeployable, String propertyName) {
        return isPropertyValueEmptyOrNull(currentDeployable, propertyName) && isPropertyValueEmptyOrNull(previousDeployable, propertyName);
    }

    private void copyFromPreviousDeployed(ConfigurationItem currentDeployed, ConfigurationItem previousDeployed, String propertyName) {
        Object propertyValue = getValueIfExists(previousDeployed, propertyName);
        currentDeployed.setProperty(propertyName, propertyValue);
    }

    @Override
    public void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                     ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, PropertyDescriptor sourcePropertyDescriptor) {
        String propertyName = targetPropertyDescriptor.getName();
        if (sourcePropertyDescriptor != null && !isPropertyValueEmptyOrNull(currentSource, propertyName)) {
            logger.trace("Resolving '{}' value from currentSource and setting the value on the current currentTarget", propertyName);
            resolveFromSource(currentSource, currentTarget, sourcePropertyDescriptor, targetPropertyDescriptor, dictionary);
        } else if (dictionary.containsKey(targetPropertyDescriptor.getFqn())) {
            logger.trace("Resolving {} from dictionary as dictionary contains a type default for the property of the currentTarget", propertyName);
            resolveFromDictionary(currentTarget, dictionary, targetPropertyDescriptor, sourcePropertyDescriptor);
        } else if (!isPropertyValueEmptyOrNull(currentTarget, propertyName)) {
            logger.trace("Using type system default as currentTarget '{}' has non-empty default value", propertyName);
        } else {
            if (sourcePropertyDescriptor == null || isPropertyOnPreviousAndCurrentDeployableEmptyOrNull(currentSource, previousSource, propertyName)) {
                logger.trace("Copy previousTarget '{}' value to current currentTarget value", propertyName);
                copyFromPreviousDeployed(currentTarget, previousTarget, propertyName);
            } else {
                logger.trace("Leave currentTarget '{}' empty", propertyName);
            }
        }
    }
}
