package com.xebialabs.deployit.filter

import ai.digital.configuration.central.deploy.ClientProperties
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.config.SameSiteHelper
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import jakarta.servlet.FilterChain
import jakarta.servlet.http.{Cookie, HttpServletRequest, HttpServletResponse}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.web.csrf.CsrfToken
import org.springframework.web.filter.OncePerRequestFilter
import org.springframework.web.util.WebUtils

class CsrfHeaderFilter(@Autowired var clientConfiguration: ClientProperties) extends OncePerRequestFilter {
  val DEFAULT_CSRF_COOKIE_NAME = "XSRF-TOKEN"

  val DEFAULT_CSRF_HEADER_NAME = "X-XSRF-TOKEN"

  val cookieName: String = DEFAULT_CSRF_COOKIE_NAME

  val serverConfiguration: ServerConfiguration = ServerConfiguration.getInstance

  override protected def doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain): Unit = {
    val csrf = request.getAttribute(classOf[CsrfToken].getName) match {
      case c: CsrfToken if c != null => c
      case other => throw new DeployitException(s"Expected CsrfToken non null - but extracted ${other.getClass.getSimpleName} - $other")
    }
    val cookie = WebUtils.getCookie(request, cookieName)
    val token = csrf.getToken
    if (cookie == null || token != null && !(token == cookie.getValue)) {
      val newCookie = new Cookie(cookieName, token)
      newCookie.setPath(getRequestContext(request))
      if (serverConfiguration.isSsl) {
        newCookie.setSecure(true)
      }
      newCookie.setAttribute("SameSite", SameSiteHelper.getStringValue(clientConfiguration.getSameSiteCookie, ClientProperties.DEFAULT_SAMESITE_COOKIE))
      response.addCookie(newCookie)
    }
    filterChain.doFilter(request, response)
  }

  private def getRequestContext(request: HttpServletRequest): String = {
    val contextPath = request.getContextPath
    if (contextPath.nonEmpty) {
      contextPath
    }
    else {
      "/"
    }
  }
}
