package com.xebialabs.deployit.service.deployment;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.Dictionaries;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class DeploymentContext {

    private final DeployedApplication deployedApplication;
    private final Dictionaries dictionaries;
    private final BeanSupplier beanSupplier;
    private final DeployedTypeSupplier typeSupplier;
    private final ExistingDeployedSupplier existingDeployedSupplier;
    private final DeployedIdGenerator deployedIdGenerator;
    private final Map<String, ConsolidatedDictionary> containerDictionaryCache = new HashMap<>();

    public DeploymentContext(DeployedApplication deployedApplication, Dictionaries dictionaries, BeanSupplier beanSupplier, DeployedTypeSupplier typeSupplier, ExistingDeployedSupplier existingDeployedSupplier) {
        this(deployedApplication, dictionaries, beanSupplier, typeSupplier, existingDeployedSupplier, IdGenerator::generateId);
    }

    public DeploymentContext(DeployedApplication deployedApplication, Dictionaries dictionaries, BeanSupplier beanSupplier, DeployedTypeSupplier typeSupplier, ExistingDeployedSupplier existingDeployedSupplier, DeployedIdGenerator deployedIdGenerator) {
        this.deployedApplication = deployedApplication;
        this.dictionaries = dictionaries;
        this.beanSupplier = beanSupplier;
        this.typeSupplier = typeSupplier;
        this.existingDeployedSupplier = existingDeployedSupplier;
        this.deployedIdGenerator = deployedIdGenerator;
    }

    public DeployedTypeSupplier getTypeSupplier() {
        return typeSupplier;
    }

    public Dictionaries getDictionaries() {
        return dictionaries;
    }

    public ConsolidatedDictionary dictionaryFilteredByContainer(Container container) {
        return containerDictionaryCache.computeIfAbsent(container.getId(), k -> dictionaries.filterBy(container).consolidate());
    }

    public ExistingDeployedSupplier getExistingDeployedSupplier() {
        return existingDeployedSupplier;
    }

    public BeanSupplier getBeanSupplier() {
        return beanSupplier;
    }

    public DeployedApplication getDeployedApplication() {
        return deployedApplication;
    }

    public DeployedIdGenerator getDeployedIdGenerator() {
        return deployedIdGenerator;
    }

    public interface BeanSupplier {
        <T> T getBean(Class<T> beanClass);
    }

    public interface DeployedTypeSupplier {
        Type getType(Deployable deployable, Container container);
    }

    public interface ExistingDeployedSupplier {
        List<Deployed> getExistingDeployed(Deployable deployable);

        default Optional<Deployed> getExistingDeployedWithContainer(Deployable deployable, Container container) {
            List<Deployed> existingDeployeds = getExistingDeployed(deployable);
            return existingDeployeds == null ? Optional.empty() : existingDeployeds.stream().filter(d -> d.getContainer().equals(container)).findFirst();
        }
    }

    public interface DeployedIdGenerator {
        String generatedId(Container container, Deployable deployable);
    }

}

