package com.xebialabs.deployit.service.importer.source;

import com.google.common.io.Files;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.util.GuavaFiles;
import org.apache.http.*;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import static com.google.common.base.Strings.emptyToNull;

public class UrlSource implements ImportSource {

    private static final Logger logger = LoggerFactory.getLogger(UrlSource.class);

    private static final String CONTENT_DISPOSITION = "Content-disposition";
    private static final String ATTACHMENT = "attachment";
    private static final String FILENAME = "filename";

    private final URL location;
    private final String user;
    private final String password;
    private FileSource downloaded;
    private File tempDir;

    public UrlSource(URL location, String user, String password) {
        this.location = location;
        this.user = user;
        this.password = password;
    }

    @Override
    public File getFile() {
        if (downloaded == null) {
            download();
        }

        return downloaded.getFile();
    }

    @Override
    public void cleanUp() {
        if (downloaded != null) {
            downloaded.cleanUp();
        }
        if (tempDir != null) {
            GuavaFiles.deleteQuietly(tempDir);
        }
    }

    private void download() {
        try (CloseableHttpClient client = HttpClients.createSystem()) {
            HttpClientContext context = configureAuthentication();
            logger.debug("Preparing to download package from {}", location);
            URI uri = location.toURI();
            HttpResponse response = executeGetRequest(client, context, uri);
            URI finalURI = getFinalUri(context, uri);
            String fileName = getFileName(response, finalURI);
            saveFile(response, fileName);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    protected URI getFinalUri(HttpClientContext context, URI uri) throws URISyntaxException {
        return URIUtils.resolve(uri, context.getTargetHost(), context.getRedirectLocations());
    }

    protected HttpResponse executeGetRequest(HttpClient client, HttpContext context, URI uri) throws IOException {
        HttpGet get = new HttpGet(uri);
        HttpResponse response = client.execute(get, context);
        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode != HttpStatus.SC_OK) {
            throw new RuntimeIOException("Failed to download package,status="+ statusCode +", from url " + location);
        }
        return response;
    }

    private HttpClientContext configureAuthentication() {
        HttpClientContext context = HttpClientContext.create();
        if (user != null) {
            BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
            credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(user, password));
            context.setCredentialsProvider(credentialsProvider);
            BasicAuthCache authCache = new BasicAuthCache();
            authCache.put(
                    new HttpHost(location.getHost(), location.getPort(), location.getProtocol()),
                    new BasicScheme()
            );
            context.setAuthCache(authCache);
        }
        return context;
    }

    private void saveFile(HttpResponse response, String fileName) throws IOException {
        try(InputStream in = response.getEntity().getContent()) {
            tempDir = Files.createTempDir();
            File archive = new File(tempDir, fileName);
            Files.asByteSink(archive).writeFrom(in);
            this.downloaded = new FileSource(archive, true);
            logger.debug("Successfully downloaded file {}", downloaded.getFile().getName());
        }
    }

    private String getFileName(HttpResponse response, URI finalURI) {
        String filename = getFilenameFromHeader(response.getLastHeader(CONTENT_DISPOSITION));
        if (filename == null) {
            logger.trace("File name could not be resolved through 'Content-disposition' header");
            filename = getFileNameFromPath(finalURI.getPath());
        }
        logger.debug("File name resolved as {}", filename);
        return filename;
    }

    private String getFilenameFromHeader(Header contentDisposition) {
        // check the Content-disposition header for a "filename" param
        if (contentDisposition != null) {
            for (HeaderElement element : contentDisposition.getElements()) {
                if (element.getName().equalsIgnoreCase(ATTACHMENT)) {
                    NameValuePair filenameParam = element.getParameterByName(FILENAME);
                    return (filenameParam != null) ? emptyToNull(filenameParam.getValue()) : null;
                }
            }
        }
        return null;
    }

    private String getFileNameFromPath(final String path) {
        logger.trace("Getting target file name from path {}", path);
        return new File(path).getName();
    }

    @Override
    public String toString() {
        return "UrlSource[" + location + "]";
    }
}
