package com.xebialabs.deployit.core.rest

import java.util.stream

import jakarta.ws.rs.container._
import jakarta.ws.rs.core.FeatureContext
import jakarta.ws.rs.ext.{Provider, WriterInterceptor, WriterInterceptorContext}
import org.springframework.transaction.support.DefaultTransactionDefinition
import org.springframework.transaction.{PlatformTransactionManager, TransactionStatus}

@Provider
class StreamingFeature (transactionManager: PlatformTransactionManager) extends DynamicFeature {
  val requestFilter = new StreamingRequestFilter(transactionManager)
  val outputWriter = new StreamingOutputWriter(transactionManager)

  override def configure(resourceInfo: ResourceInfo, context: FeatureContext): Unit = {
    if (classOf[stream.Stream[_]].isAssignableFrom(resourceInfo.getResourceMethod.getReturnType)) {
      context.register(requestFilter, classOf[ContainerRequestFilter])
      context.register(outputWriter, classOf[WriterInterceptor])
    }
  }
}

@Provider
class StreamingRequestFilter (transactionManager: PlatformTransactionManager) extends ContainerRequestFilter {

  override def filter(requestContext: ContainerRequestContext): Unit = {
    val transactionStatus = transactionManager.getTransaction(transactionDefinition)
    requestContext.setProperty(StreamingFilter.TRANSACTION_NAME, transactionStatus)
  }

  private def transactionDefinition = {
    val transactionDefinition = new DefaultTransactionDefinition()
    transactionDefinition.setReadOnly(true)
    transactionDefinition
  }

}

@Provider
class StreamingOutputWriter (transactionManager: PlatformTransactionManager) extends WriterInterceptor {

  override def aroundWriteTo(context: WriterInterceptorContext):Unit = {
    try {
      context.proceed()
    } finally {
      val transactionStatus = context.getProperty(StreamingFilter.TRANSACTION_NAME).asInstanceOf[TransactionStatus]
      if (!transactionStatus.isCompleted) transactionManager.rollback(transactionStatus)
    }
  }
}

object StreamingFilter {
  val TRANSACTION_NAME = "STREAMING_TRANSACTION"
}
