package com.xebialabs.deployit.service.dependency


import scalax.collection.GraphPredef.EdgeLikeIn
import scalax.collection.constrained
import scalax.collection.constrained._
import scalax.collection.constrained.constraints.{Acyclic, NoneConstraint}
import scalax.collection.edge.LDiEdge

object DependencyConstraints {

  private[dependency] def validateApplicationGraph(dependencies: ApplicationGraph): Unit = {
    implicit val constraints: Config = AcyclicApps && UnresolvedApplication && UnresolvedVersion && AmbiguousVersion
    dependencies.foldLeft(constrained.Graph.empty[Application, LDiEdge])(_ + _)
  }

  class DependencyException(msg: String) extends IllegalArgumentException(msg)

  object AcyclicApps extends ConstraintCompanion[Acyclic] {
    override def apply[N, E[+X] <: EdgeLikeIn[X], G <: Graph[N, E]](self: G): Acyclic[N, E, G] = new Acyclic[N, E, G](self) {
      var success: Boolean = true

      override def preAdd(edge: E[N]): PreCheckResult = {
        val preCheckResult = super.preAdd(edge)
        success = !preCheckResult.abort
        PreCheckResult.apply(PreCheckFollowUp.PostCheck)
      }

      override def postAdd(newGraph: G, passedNodes: Iterable[N], passedEdges: Iterable[E[N]], preCheck: PreCheckResult): Either[PostCheckFailure, G] = {
        val result = super.postAdd(newGraph, passedNodes, passedEdges, preCheck)
        if (success) {
          success = result.isRight
        }

        if (!success) {
          val cycle = newGraph.findCycle.map(_.edges.map(_.toOuter).map(e => s"${e._1} to ${e._2}:${e.label}").mkString(", "))
          throw new DependencyException(s"Error while resolving dependencies. There is a cyclic dependency in the chain. Cause: [${cycle.getOrElse("dependency on itself")}]")
        }
        result
      }
    }
  }

  object UnresolvedApplication extends ConstraintCompanion[NoneConstraint] {
    override def apply[N, E[+X] <: EdgeLikeIn[X], G <: Graph[N, E]](self: G): NoneConstraint[N, E, G] = new NoneConstraint[N, E, G](self) {
      override def preAdd(edge: E[N]): PreCheckResult = {
        if (edge._2.isInstanceOf[UnresolvedDependency]) {
          throw new DependencyException(s"""Error while trying to resolve the dependencies of application "${edge._1}". Cannot find an application with the name "${edge._2}"""")
        }

        PreCheckResult.complete(true)
      }
    }
  }

  object UnresolvedVersion extends ConstraintCompanion[NoneConstraint] {
    override def apply[N, E[+X] <: EdgeLikeIn[X], G <: Graph[N, E]](self: G): NoneConstraint[N, E, G] = new NoneConstraint[N, E, G](self) {
      override def preAdd(edge: E[N]): PreCheckResult = {
        if (edge.label.asInstanceOf[Label].udmVersion.isEmpty) {
          throw new DependencyException(s"""Error while trying to resolve the dependencies of application "${edge._1}". Cannot find matching version of application "${edge._2}" for version range '${edge.label.asInstanceOf[Label].range}'""")
        }
        PreCheckResult.complete(true)
      }
    }
  }

  object AmbiguousVersion extends ConstraintCompanion[NoneConstraint] {
    override def apply[N, E[+X] <: EdgeLikeIn[X], G <: Graph[N, E]](self: G): NoneConstraint[N, E, G] = new NoneConstraint[N, E, G](self) {
      override def preAdd(edge: E[N]): PreCheckResult = PreCheckResult.postCheck(true)

      override def postAdd(newGraph: G, passedNodes: Iterable[N], passedEdges: Iterable[E[N]], preCheck: PreCheckResult): Either[PostCheckFailure, G] = {
        if (passedEdges.size == 1) {
          val application = passedEdges.headOption.
            getOrElse(throw new NoSuchElementException("Can't read passedEdges head value"))._2
          val incomingEdges = newGraph.get(application).incoming
          val success = incomingEdges.groupBy(_.label).size == 1

          if (!success) {
            val invalidReferences = incomingEdges.map { edge =>
              s"${edge.toOuter._1} (needs ${edge.label})"
            }.mkString(", ")
            val msg = s"""Error while trying to resolve dependencies. There are multiple dependencies to application "$application", but there is no version that satisfies all version ranges. $invalidReferences"""
            throw new DependencyException(msg)
          }
        }
        Right(newGraph)
      }
    }
  }

}
