package com.xebialabs.deployit.security.principaldata;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Factory bean responsible for creating a PrincipalDataProvider instance.
 * This class integrates with Spring's lifecycle to configure and provide a PrincipalDataProvider
 * based on the application context's available resources.
 */
@Service
public class PrincipalDataProviderFactoryBean implements FactoryBean<PrincipalDataProvider>, ApplicationContextAware, InitializingBean {
    private static final Logger logger = LoggerFactory.getLogger(PrincipalDataProviderFactoryBean.class);

    private ApplicationContext context;

    private PrincipalDataProvider provider;

    /**
     * Initializes the PrincipalDataProvider after the properties are set.
     * This method configures the provider with available user data providers, prioritizing local, crowd, and LDAP sources.
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        PrincipalDataProviderImpl principalDataProvider = new PrincipalDataProviderImpl();

        // 1. Local user data provider has highest priority

        // 2. Crowd user data provider

        // 3. LDAP user data provider
        List<UserDataProvider> ldapUserDataProviderList = tryInitUserDataProviderFromSpringLdapProvider();
        if (!ldapUserDataProviderList.isEmpty()) {
            principalDataProvider.addUserProvider(ldapUserDataProviderList);
        }

        // 4. Other data providers

        this.provider = principalDataProvider;
    }

    @Override
    public PrincipalDataProvider getObject() throws Exception {
        return provider;
    }

    @Override
    public Class<?> getObjectType() {
        return PrincipalDataProvider.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    /**
     * Tries to obtain an LDAP connection from a "standard" Spring Security configuration (as described in the XLDeploy
     * configuration manual).
     * <p>
     * Due to the Spring implementation, this requires some reflection.
     */
    private List<UserDataProvider> tryInitUserDataProviderFromSpringLdapProvider() {
        Map<String, LdapAuthenticationProvider> ldapProviders =
                context.getBeansOfType(LdapAuthenticationProvider.class);

        List<UserDataProvider> ldapUserDataProviderList = new ArrayList<>();
        if (!ldapProviders.isEmpty()) {
            try {
                logger.info("Found {} in spring context, initializing from LDAP", LdapAuthenticationProvider.class.getSimpleName());
                ldapProviders.forEach((name, ldapProvider) -> ldapUserDataProviderList.add(new LdapUserDataProvider(ldapProvider)));
            } catch (Exception e) {
                logger.warn("Error initializing from {}", LdapAuthenticationProvider.class.getSimpleName(), e);
            }
        }
        return ldapUserDataProviderList;
    }

    /**
     * Sets the ApplicationContext that this object runs in.
     * This method is called by the Spring container to provide the ApplicationContext.
     *
     * @param context the ApplicationContext object to be used by this object
     * @throws BeansException if thrown by application context methods
     */
    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        this.context = context;
    }
}
