package com.xebialabs.deployit.security;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterators.addAll;
import static com.google.common.collect.Iterators.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.jcr.JcrConstants.SECURITY_NODE_ID;

import java.io.IOException;
import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.security.auth.Subject;

import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.principal.PrincipalManager;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.security.principal.AdminPrincipal;
import org.apache.jackrabbit.value.StringValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PermissionHelper;
import com.xebialabs.deployit.util.Tuple;


@Component("permissionService")
public class  JcrPermissionService implements PermissionService {

	protected JcrTemplate jcrTemplate;

	private Function<Value, String> valueToStringFunction = new Function<Value, String>() {
			@Override
			public String apply(Value input) {
				try {
					return input.getString();
				} catch (RepositoryException e) {
					throw new IllegalStateException("Something wrong with the security in the repository; Could not get String from value " + input, e);
				}
			}
		};

	@Autowired
    public JcrPermissionService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

	@Override
	public Map<String, Set<String>> getUserPermissions(final String principalName) {
		return jcrTemplate.execute(new JcrCallback<Map<String, Set<String>>>() {
			@Override
			public Map<String, Set<String>> doInJcr(Session session) throws IOException, RepositoryException {
				Map<String, Set<String>> userPermissions = new TreeMap<String, Set<String>>();
				final Node securityNode = checkNotNull(session.getNode(SECURITY_NODE_ID));
				final PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
				Principal principal = principalManager.getPrincipal(session.getUserID());
				if (principal.getName().equals(principalName) || (principal instanceof AdminPrincipal)) {
					List<String> groups = getGroupMemberships(principalManager, principalManager.getPrincipal(principalName));
					for (String permission : getAllowedPermissions(securityNode, groups, principalName)) {
						Tuple<Permission, String> deconstructedPermission = PermissionHelper.decodePermissionString(permission);
						String permissionName = deconstructedPermission.a.getPermissionName();
						if (!userPermissions.containsKey(permissionName)) {
							userPermissions.put(permissionName, Sets.<String>newTreeSet());
						}
						if (deconstructedPermission.b.length() == 0) {
							userPermissions.get(permissionName).add("");
						} else {
							userPermissions.get(permissionName).add(deconstructedPermission.b.replace("$", "/"));
						}
					}					
				} else {
					throw new RepositoryException("Insufficient permissions for retrieving a user's or group's permissions");
				}
				return userPermissions;
			}
		});
	}

    @Override
	public boolean hasUserPermission(final String permissionName, final String principalName) {
		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				final PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
				final Principal principal = principalManager.getPrincipal(principalName);
				if (isAdmin(principal, permissionName)) return true;
				List<String> groups = getGroupMemberships(principalManager, principal);

				return checkHasPermission(session, groups, principalName, permissionName);
			}
		});
	}

	@Override
	public boolean hasLoggedInUserPermission(final String permissionName) {
		final Authentication auth = SecurityContextHolder.getContext().getAuthentication();

		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				// N.B.: Casting the session to SessionImpl is the only way to get the Subject.
				final Subject subject = ((SessionImpl) session).getSubject();
				final PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
				final List<String> groups = newArrayList();
				for (Principal eachPrincipal : subject.getPrincipals()) {
					if (isAdmin(eachPrincipal, permissionName)) return true;
					groups.addAll(getGroupMemberships(principalManager, eachPrincipal));
				}

				final String principalName = auth.getName();

				return checkHasPermission(session, groups, principalName, permissionName);
			}
		});
	}

	private boolean isAdmin(Principal eachPrincipal, String permissionName) {
		if (eachPrincipal instanceof AdminPrincipal) {
			logger.debug("Granted {} because user is admin.", permissionName);
			return true;
		}
		return false;
	}

	private Boolean checkHasPermission(Session session, List<String> groups, String principalName, String permissionName) throws RepositoryException {
		logger.debug("Checking permission for principal [{}] with groups [{}]", principalName, groups);
		final Node securityNode = checkNotNull(session.getNode(SECURITY_NODE_ID));
		Set<String> permissions = getAllowedPermissions(securityNode, groups, principalName);
		logger.trace("Checking for permission [{}] in allowed permissions: [{}]", permissionName, permissions);
		return permissions.contains(permissionName);
	}

	private Set<String> getAllowedPermissions(Node securityNode, List<String> groups, String principalName) {
		Set<String> permissions = newHashSet();
		addAllPermissionsToSet(principalName, permissions, securityNode);
		for (String group : groups) {
			addAllPermissionsToSet(group, permissions, securityNode);
		}

		logger.trace("All permissions for [{},{}] are [{}]", new Object[]{principalName, groups, permissions});
		return permissions;
	}

	private void addAllPermissionsToSet(String principal, Set<String> permissions, Node securityNode) {
		try {
			Property node = securityNode.getProperty(principal);
			permissions.addAll(Lists.transform(newArrayList(node.getValues()), valueToStringFunction));
		} catch (RepositoryException e) {
			// Ignore, group has no set permissions.
		}
	}

	@Override
	public void grantPermission(final String permissionName, final String principal) {
		logger.debug("Granting {} to {}", permissionName, principal);
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				Node securityNode = checkNotNull(session.getNode(SECURITY_NODE_ID));

				Set<String> permissions = getAllowedPermissions(securityNode, Lists.<String>newArrayList(), principal);
				permissions.add(permissionName);
				setPermissionsForPrincipal(securityNode, principal, permissions);

				session.save();
				return null;
			}
		});
	}

	@Override
	public void denyPermission(final String permissionName, final String principal) {
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				Node securityNode = checkNotNull(session.getNode(SECURITY_NODE_ID));

				Set<String> permissions = getAllowedPermissions(securityNode, Lists.<String>newArrayList(), principal);
				permissions.remove(permissionName);
				setPermissionsForPrincipal(securityNode, principal, permissions);

				session.save();
				return null;
			}
		});
	}

	private void setPermissionsForPrincipal(Node securityNode, String principal, Set<String> permissions) throws RepositoryException {
		securityNode.setProperty(principal, Lists.transform(newArrayList(permissions), new Function<String, Value>() {
			@Override
			public Value apply(String input) {
				return new StringValue(input);
			}
		}).toArray(new Value[permissions.size()]));
	}

	private static final Logger logger = LoggerFactory.getLogger(JcrPermissionService.class);

	@SuppressWarnings("unchecked")
	private List<String> getGroupMemberships(PrincipalManager principalManager, Principal principal) {
		final List<String> groups = newArrayList();
		if (principal != null) { 
			addAll(groups, transform(principalManager.getGroupMembership(principal), new Function<Principal, String>() {
				@Override
				public String apply(Principal input) {
					return input.getName();
				}
			}));
		}
		return groups;
	}

}
