package com.xebialabs.deployit.security.permission;

import static com.google.common.collect.Lists.newArrayList;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.security.SecurityServiceLocator;
import com.xebialabs.deployit.util.Tuple;

public class PermissionHelper {
	static void grantPermission(Permission permission, String user, String onConfigurationItem) {
		grantPermission(encodePermissionString(permission, onConfigurationItem), user);
	}

	static void grantPermission(Permission permission, String user) {
		grantPermission(permission.getPermissionName(), user);
	}

	private static void grantPermission(String permission, String user) {
		SecurityServiceLocator.getPermissionService().grantPermission(permission, user);
	}

	static void revokePermission(Permission permission, String user, String onConfigurationItem) {
		revokePermission(encodePermissionString(permission, onConfigurationItem), user);
	}

	static void revokePermission(Permission permission, String user) {
		revokePermission(permission.getPermissionName(), user);
	}

	private static void revokePermission(String permission, String user) {
		SecurityServiceLocator.getPermissionService().denyPermission(permission, user);
	}

	static boolean hasPermission(Permission permission) {
		return hasPermission(permission.getPermissionName());
	}

	private static boolean hasPermission(String permission) {
		return SecurityServiceLocator.getPermissionService().hasLoggedInUserPermission(permission);
	}

	static boolean hasPermission(Permission permission, List<String> onConfigurationItems) {
		boolean allowed = !onConfigurationItems.isEmpty();
		for (String onConfigurationItem : onConfigurationItems) {
            List<String> fullTree = getFullTreeAsSeparateNodes(onConfigurationItem);
            boolean hasPermissionOnTreePart = false;
            for (String s : fullTree) {
                hasPermissionOnTreePart = hasPermissionOnTreePart || hasPermission(permission, s);
            }
            allowed = allowed && hasPermissionOnTreePart;
		}
		return allowed || hasPermission(permission);
	}

    private static List<String> getFullTreeAsSeparateNodes(String onConfigurationItem) {
        ArrayList<String> list = newArrayList();
        list.add(onConfigurationItem);
        for (int index = onConfigurationItem.indexOf('/'); index > -1; index = onConfigurationItem.indexOf('/', index + 1)) {
            list.add(onConfigurationItem.substring(0, index));
        }

        logger.debug("Split off tree: {} to {}", onConfigurationItem, list);

        return list;
    }

    private static boolean hasPermission(Permission permission, String onConfigurationItem) {
		return hasPermission(encodePermissionString(permission, onConfigurationItem));
	}

	static boolean hasPermission(Permission permission, String principal, String onConfigurationItem) {
		return SecurityServiceLocator.getPermissionService().hasUserPermission(encodePermissionString(permission, onConfigurationItem), principal);
	}

	public static String encodePermissionString(Permission permission, String onConfigurationItem) {
		final StringBuilder builder = new StringBuilder(permission.getPermissionName());
		if (onConfigurationItem != null) {
			logger.debug("constructing permission: {} with {} ", builder.toString(), onConfigurationItem);
			builder.append("_").append(onConfigurationItem.replaceAll("/", "\\$"));
		}
		return builder.toString();
	}

	public static Tuple<Permission, String> decodePermissionString(String permissionString) {
		for (Permission permission : Permission.values()) {
			if (permissionString.startsWith(permission.getPermissionName())) {
				return decodePermissionString(permissionString, permission);
			}
		}
		throw new RuntimeException("PermissionString " + permissionString + " could not be decoded");
	}

	private static Tuple<Permission, String> decodePermissionString(String permissionString, Permission permission) {
		int permissionNameCiSeparatorIndex = permission.getPermissionName().length() + 1;
		if (permissionNameCiSeparatorIndex >= permissionString.length()) {
			return new Tuple<Permission, String>(permission, "");
		}
		String ciName = permissionString.substring(permissionNameCiSeparatorIndex).replace("$", "/");
		return new Tuple<Permission, String>(permission, ciName);
	}

	private static final Logger logger = LoggerFactory.getLogger(PermissionHelper.class);
}
