package com.xebialabs.deployit.service.deployment;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.deployment.planner.Planner;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.execution.Plan;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.deployment.InitialDeploymentTask;
import com.xebialabs.deployit.task.deployment.UndeploymentTask;
import com.xebialabs.deployit.task.deployment.UpgradeDeploymentTask;
import com.xebialabs.deployit.task.step.RepositoryUpdateStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

@Component
public class DeploymentService {

    private RepositoryService repositoryService;

    private Planner planner;


    @Autowired
    public DeploymentService(RepositoryService repositoryService, Planner planner) {
        this.repositoryService = repositoryService;
        this.planner = planner;
    }

    @SuppressWarnings("rawtypes")
    public Task prepareInitialDeployment(final Version pkg, final Environment env, final Collection<Deployed> deployedPojos, WorkDir workDir) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().initial(pkg, env);

        DeploymentOperationCalculator.calculate(builder, Sets.<Deployed>newHashSet(), newHashSet(deployedPojos));

        final DeltaSpecification specification = builder.build();
        final Plan plan = planner.plan(specification);
	    List<DeploymentStep> steps = getStepsFromPlanOrDefaultUpdateStep(plan);
	    return new InitialDeploymentTask(specification, steps, repositoryService, workDir);
    }

    @SuppressWarnings("rawtypes")
    public Task prepareUpgradeDeployment(final Version newPkg, final DeployedApplication existingDeployment, final Collection<Deployed> deployeds, WorkDir oldWorkDir, WorkDir newWorkDir) {
        final DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().upgrade(newPkg, existingDeployment);
        final Set<Deployed> incomingDeployedPojos = newHashSet(deployeds);

	    logger.trace("Incoming Deployeds {}", incomingDeployedPojos);
	    logger.trace("Existing Deployeds {}", existingDeployment.getDeployeds());

        DeploymentOperationCalculator.calculate(builder, existingDeployment.getDeployeds(), incomingDeployedPojos);

        final DeltaSpecification specification = builder.build();
        final Plan plan = planner.plan(specification);
	    List<DeploymentStep> steps = getStepsFromPlanOrDefaultUpdateStep(plan);
        return new UpgradeDeploymentTask(specification, existingDeployment, steps, repositoryService, oldWorkDir, newWorkDir);
    }

    @SuppressWarnings("rawtypes")
    public Task prepareUndeployment(final DeployedApplication deployedApplication, WorkDir workDir) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().undeploy(deployedApplication);
        
        DeploymentOperationCalculator.calculate(builder, deployedApplication.getDeployeds(), Sets.<Deployed>newHashSet());

        final DeltaSpecification specification = builder.build();
        final Plan plan = planner.plan(specification);
	    List<DeploymentStep> steps = getStepsFromPlanOrDefaultUpdateStep(plan);
        return new UndeploymentTask(specification, deployedApplication, steps, repositoryService, workDir);
    }

	private List<DeploymentStep> getStepsFromPlanOrDefaultUpdateStep(Plan plan) {
		List<DeploymentStep> steps = plan.getSteps();
		if (steps.isEmpty()) {
			steps = newArrayList((DeploymentStep) new RepositoryUpdateStep());
		}
		return steps;
	}

	@SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class DeploymentException extends DeployitException {
        public DeploymentException(final String message) {
            super(message);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);
}
