package com.xebialabs.deployit.task.jcrarchive;

import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.STEP_NODE_NAME_PREFIX;
import static com.xebialabs.deployit.jcr.JcrConstants.TASKS_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;

import java.io.IOException;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.query.qom.QueryObjectModel;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.grouping.GroupBy;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTask;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.TaskStep;
import com.xebialabs.deployit.task.TaskStepInfo;

@Component
public class JcrTaskArchive implements TaskArchive {

	static final String STATE = "state";
	static final String START_DATE = "startDate";
	static final String COMPLETION_DATE = "completionDate";
	static final String LAST_MODIFICATION_DATE = "lastModificationDate";
	static final String NR_OF_STEPS = "nrOfSteps";
	static final String CURRENT_STEP_NR = "currentStepNr";
	static final String OWNING_USER = "ownerUser";
	static final String FAILURE_COUNT = "failureCount";
	static final String LOG = "log";
	static final String DESCRIPTION = "description";
	static final String APPLICATION = "application";
	static final String VERSION = "version";
	static final String ENVIRONMENT = "environment";
	static final String DEPLOYMENT_TYPE = "deploymentType";

	private final JcrTemplate jcrTemplate;

	private static final Pattern qualifiedTaskIdPattern = Pattern.compile("(.*)/(.*)/(.*)/(.*)"); // ie. env/app/version/uuid

	@Autowired
	public JcrTaskArchive(final JcrTemplate jcrTemplate) {
		this.jcrTemplate = jcrTemplate;
	}

	@Override
	public void archiveTask(final DeploymentTask task) {
		checkAllMandatoryAttributesArePresentInDeploymentTask(task);
		saveDeploymentTask(task);
	}

	private void checkAllMandatoryAttributesArePresentInDeploymentTask(DeploymentTask task) {
		checkState(!Strings.nullToEmpty(task.getApplicationName()).isEmpty(), "applicationName in deployment task must be set");
		checkState(!Strings.nullToEmpty(task.getApplicationVersion()).isEmpty(), "applicationVersion in deployment task must be set");
		checkState(!Strings.nullToEmpty(task.getEnvironment()).isEmpty(), "environment in deployment task must be set");
	}

	private void saveDeploymentTask(final DeploymentTask task) {
		jcrTemplate.execute(new JcrCallback<String>() {
			@Override
			public String doInJcr(Session session) throws IOException, RepositoryException {
				Node jcrTaskNode = createJcrTaskNodeAndAssociatedParentNodes(task, getTasksRootNode(session));
				mapDeploymentTaskToJcrNode(task, jcrTaskNode);
				session.save();
				return null;
			}
		});
	}

	private Node createJcrTaskNodeAndAssociatedParentNodes(final DeploymentTask task, Node tasksRootNode) throws RepositoryException {
		Node envNode = createNodeIfDoesNotExist(tasksRootNode, task.getEnvironment());
		Node appNode = createNodeIfDoesNotExist(envNode, task.getApplicationName());
		Node verNode = createNodeIfDoesNotExist(appNode, task.getApplicationVersion());
		Node node = verNode.addNode(task.getId());
		node.addMixin(TASK_NODETYPE_NAME);
		return node;
	}

	private Node createNodeIfDoesNotExist(Node parentNode, String nodeId) throws RepositoryException {
		if (parentNode.hasNode(nodeId)) {
			return parentNode.getNode(nodeId);
		} else {
			return parentNode.addNode(nodeId);
		}
	}

	private void mapDeploymentTaskToJcrNode(final DeploymentTask task, Node taskNode) throws IOException, RepositoryException {
		taskNode.setProperty(ID_PROPERTY_NAME, task.getId());
		taskNode.setProperty(STATE, task.getState().name());
		taskNode.setProperty(START_DATE, task.getStartDate());
		taskNode.setProperty(COMPLETION_DATE, task.getCompletionDate());
		taskNode.setProperty(CURRENT_STEP_NR, task.getCurrentStepNr());
		taskNode.setProperty(NR_OF_STEPS, task.getNrOfSteps());
		taskNode.setProperty(OWNING_USER, task.getOwner());
		taskNode.setProperty(APPLICATION, task.getApplicationName());
		taskNode.setProperty(VERSION, task.getApplicationVersion());
		taskNode.setProperty(ENVIRONMENT, task.getEnvironment());
		taskNode.setProperty(FAILURE_COUNT, task.getFailureCount());
		taskNode.setProperty(DEPLOYMENT_TYPE, task.getDeploymentType().name());

		for (int nr = 1; nr <= task.getNrOfSteps(); nr++) {
			TaskStep step = task.getStep(nr);
			Node stepNode = taskNode.addNode(STEP_NODE_NAME_PREFIX + nr);
			stepNode.setProperty(DESCRIPTION, step.getDescription());
			stepNode.setProperty(LAST_MODIFICATION_DATE, step.getLastModificationDate());
			stepNode.setProperty(STATE, step.getState().name());
			stepNode.setProperty(START_DATE, step.getStartDate());
			stepNode.setProperty(COMPLETION_DATE, step.getCompletionDate());
			stepNode.setProperty(FAILURE_COUNT, step.getFailureCount());
			stepNode.setProperty(LOG, step.getLog());
		}
	}

	@Override
	public DeploymentTaskInfo getTask(final String taskId) {
		if (qualifiedTaskIdPattern.matcher(taskId).matches()) {
			return getTaskUsingFullyQualifiedPath(taskId);
		} else {
			return getTaskUsingUuid(taskId);
		}

	}

	private DeploymentTaskInfo getTaskUsingFullyQualifiedPath(final String fullyQualifiedPath) {
		return jcrTemplate.execute(new JcrCallback<DeploymentTaskInfo>() {
			@Override
			public DeploymentTaskInfo doInJcr(Session session) throws IOException, RepositoryException {
				try {
					Node tasksNode = getTasksRootNode(session);
					Node taskNode = tasksNode.getNode(fullyQualifiedPath);
					DeploymentTaskInfo task = mapJcrTaskNodeToDeploymentTaskWithoutLoadingSteps(taskNode);
					if (task == null) {
						throw new NotFoundException("Cannot load task " + fullyQualifiedPath + " because that object with that id is not a task");
					}
					task.setSteps(loadTaskStepInfosForJcrTaskNode(taskNode));
					return task;
				} catch (PathNotFoundException exc) {
					throw new NotFoundException("Cannot load task " + fullyQualifiedPath + ": " + exc.toString(), exc);
				}
			}
		});
	}

	private DeploymentTaskInfo getTaskUsingUuid(String taskUuid) {
		ArchivedTaskSearchParameters taskSearch = new ArchivedTaskSearchParameters();
		taskSearch.withUniqueId(taskUuid);
		Collection<DeploymentTaskInfo> deploymentTaskInfos = searchTasks(taskSearch);
		if (deploymentTaskInfos.size() == 1) {
			return deploymentTaskInfos.iterator().next();
		} else if (deploymentTaskInfos.isEmpty()) {
			throw new NotFoundException("Cannot load task " + taskUuid);
		} else {
			throw new NotFoundException("Cannot load task " + taskUuid + " because there are multiple tasks with same id. " + deploymentTaskInfos);
		}
	}

	private DeploymentTaskInfo mapJcrTaskNodeToDeploymentTaskWithoutLoadingSteps(Node taskNode) throws RepositoryException {
		if (!taskNode.isNodeType(TASK_NODETYPE_NAME)) {
			return null;
		}

		DeploymentTaskInfo task = new DeploymentTaskInfo();

		task.setId(getPropertyFromJcrNode(taskNode, ID_PROPERTY_NAME));
		task.setEnvironment(getPropertyFromJcrNode(taskNode, ENVIRONMENT));
		task.setApplicationName(getPropertyFromJcrNode(taskNode, APPLICATION));
		task.setApplicationVersion(getPropertyFromJcrNode(taskNode, VERSION));
		task.setState(Task.State.valueOf(getPropertyFromJcrNode(taskNode, STATE)));
		task.setDeploymentType(DeploymentTask.DeploymentType.valueOf(getPropertyFromJcrNode(taskNode, DEPLOYMENT_TYPE)));
		task.setStartDate(getOptionalDatePropertyFromJcrNode(taskNode, START_DATE));
		task.setCompletionDate(getOptionalDatePropertyFromJcrNode(taskNode, COMPLETION_DATE));
		task.setOwner(getOptionalPropertyFromJcrNode(taskNode, OWNING_USER));
		task.setCurrentStepNr(getIntegerPropertyFromJcrNode(taskNode, CURRENT_STEP_NR));
		task.setFailureCount(getIntegerPropertyFromJcrNode(taskNode, FAILURE_COUNT));
		return task;
	}

	private String getPropertyFromJcrNode(Node node, String propName) throws RepositoryException {
		return node.getProperty(propName).getString();
	}

	private int getIntegerPropertyFromJcrNode(Node node, String propName) throws RepositoryException {
		return (int) node.getProperty(propName).getLong();
	}

	private Calendar getOptionalDatePropertyFromJcrNode(Node node, String propName) throws RepositoryException {
		if (node.hasProperty(propName)) {
			return node.getProperty(propName).getDate();
		}
		return null;
	}

	private String getOptionalPropertyFromJcrNode(Node node, String propName) throws RepositoryException {
		if (node.hasProperty(propName)) {
			return node.getProperty(propName).getString();
		}
		return null;
	}

	private List<TaskStepInfo> loadTaskStepInfosForJcrTaskNode(final Node taskNode) throws RepositoryException {
		int nrOfSteps = getIntegerPropertyFromJcrNode(taskNode, NR_OF_STEPS);
		List<TaskStepInfo> steps = Lists.newArrayList();
		for (int nr = 1; nr <= nrOfSteps; nr++) {
			Node stepNode = taskNode.getNode(STEP_NODE_NAME_PREFIX + nr);
			steps.add(mapJcrStepNodeToTaskStepInfo(stepNode));
		}
		return steps;
	}

	private TaskStepInfo mapJcrStepNodeToTaskStepInfo(Node stepNode) throws RepositoryException {
		final String description = getPropertyFromJcrNode(stepNode, DESCRIPTION);
		final TaskStep.StepState stepState = TaskStep.StepState.valueOf(getPropertyFromJcrNode(stepNode, STATE));
		final Calendar startDate = getOptionalDatePropertyFromJcrNode(stepNode, START_DATE);
		final Calendar endDate = getOptionalDatePropertyFromJcrNode(stepNode, COMPLETION_DATE);
		final Calendar lastModificationDate = getOptionalDatePropertyFromJcrNode(stepNode, LAST_MODIFICATION_DATE);
		final String log = getPropertyFromJcrNode(stepNode, LOG);
		final int failureCount = getIntegerPropertyFromJcrNode(stepNode, FAILURE_COUNT);
		return new TaskStepInfo(description, stepState, startDate, endDate, lastModificationDate, log, failureCount);
	}

	private Node getTasksRootNode(Session session) throws RepositoryException {
		return session.getRootNode().getNode(TASKS_NODE_NAME);
	}

	@Override
	public Collection<DeploymentTaskInfo> searchTasks(final ArchivedTaskSearchParameters params) {
		return searchTasks(params, true);
	}

	@Override
	public Collection<DeploymentTaskInfo> searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params) {
		return searchTasks(params, false);
	}

	private List<DeploymentTaskInfo> searchTasks(final ArchivedTaskSearchParameters params, final boolean loadSteps) {
		return jcrTemplate.execute(new JcrCallback<List<DeploymentTaskInfo>>() {
			@Override
			public List<DeploymentTaskInfo> doInJcr(Session session) throws IOException, RepositoryException {
				QueryResult queryResult = executeJcrQuery(session, params);
				final List<DeploymentTaskInfo> tasks = Lists.newArrayList();
				mapQueryResultToDeploymentTasks(queryResult, loadSteps, new TaskCallback() {
					@Override
					public void doWithTask(DeploymentTaskInfo t) {
						tasks.add(t);
					}
				});
				return tasks;
			}
		});
	}

	public void searchTasks(final ArchivedTaskSearchParameters params, final TaskCallback callback) {
		searchTasks(params, callback, true);
	}

	public void searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params, final TaskCallback callback) {
		searchTasks(params, callback, false);
	}

	private Object searchTasks(final ArchivedTaskSearchParameters params, final TaskCallback callback, final boolean loadSteps) {
	    return jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				QueryResult queryResult = executeJcrQuery(session, params);
				mapQueryResultToDeploymentTasks(queryResult, loadSteps, callback);
				return null;
			}
		});
    }

	private QueryResult executeJcrQuery(Session session, ArchivedTaskSearchParameters params) throws RepositoryException {
		QueryManager qm = session.getWorkspace().getQueryManager();
		JcrArchivedTaskSearchQueryBuilder queryBuilder = new JcrArchivedTaskSearchQueryBuilder(qm, session.getValueFactory(), params);
		QueryObjectModel query = queryBuilder.buildQuery();
		return query.execute();
	}

	private void mapQueryResultToDeploymentTasks(QueryResult result, boolean loadSteps, TaskCallback callback) throws RepositoryException {
		final RowIterator iterator = result.getRows();
		while (iterator.hasNext()) {
			try {
				Node taskNode = iterator.nextRow().getNode(JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME);
				DeploymentTaskInfo task = mapJcrTaskNodeToDeploymentTaskWithoutLoadingSteps(taskNode);
				if (loadSteps)
					task.setSteps(loadTaskStepInfosForJcrTaskNode(taskNode));
				callback.doWithTask(task);
			} catch (RepositoryException rre) {
				// Ignore, we weren't allowed to read a node, or one of
				// it's relations.
			}
		}
	}

	@Override
	public List<String> getAllTaskIds() {
		return jcrTemplate.execute(new JcrCallback<List<String>>() {
			@Override
			public List<String> doInJcr(Session session) throws IOException, RepositoryException {
				QueryResult queryResult = executeJcrQuery(session, new ArchivedTaskSearchParameters());
				return mapQueryResultToTasksIds(queryResult);
			}
		});
	}

	private List<String> mapQueryResultToTasksIds(final QueryResult queryResult) throws RepositoryException {
		List<String> taskIds = Lists.newArrayList();
		final RowIterator iterator = queryResult.getRows();
		while (iterator.hasNext()) {
			try {
				taskIds.add(getPropertyFromJcrNode(iterator.nextRow().getNode(JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME), ID_PROPERTY_NAME));
			} catch (RepositoryException rre) {
				// Ignore, we weren't allowed to read a node, or one of
				// it's relations.
			}
		}
		return taskIds;
	}

	@SuppressWarnings("unchecked")
	public Collection<Map<String, Object>> searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params, final GroupBy groupBy) {
		return (Collection<Map<String, Object>>) jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Collection<Map<String, Object>> doInJcr(Session session) throws IOException, RepositoryException {
				RowIterator iterator = executeJcrQuery(session, params).getRows();
				while (iterator.hasNext()) {
					Node taskNode = iterator.nextRow().getNode(JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME);
					groupBy.process(mapJcrTaskNodeToDeploymentTaskWithoutLoadingSteps(taskNode));
				}
				return groupBy.getResult();
			}
		});
	}

	public interface TaskCallback {
		void doWithTask(DeploymentTaskInfo t);
	}
}
