package com.xebialabs.deployit.service.deployment;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Sets.newHashSet;

import java.io.IOException;
import java.util.Set;

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.util.FileComparisonUtils;
import com.xebialabs.overthere.local.LocalFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeploymentOperationCalculator {

	@SuppressWarnings("rawtypes")
    public static DeltaSpecificationBuilder calculate(DeltaSpecificationBuilder builder, Set<Deployed> oldDeployeds, Set<Deployed> newDeployeds) {
		checkNotNull(oldDeployeds, "Cannot calculate on null old Deployeds");
		checkNotNull(newDeployeds, "Cannot calculate on null new Deployeds");

		if (oldDeployeds.isEmpty()) {
			for (Deployed newDeployed : newDeployeds) {
                builder.create(newDeployed);
            }
		} else if (newDeployeds.isEmpty()) {
			for (Deployed oldDeployed : oldDeployeds) {
                builder.destroy(oldDeployed);
            }
		} else {
			calculateUpgrade(builder, oldDeployeds, newDeployeds);
		}

		return builder;
	}

	@SuppressWarnings("rawtypes")
    static void calculateUpgrade(DeltaSpecificationBuilder builder, Set<Deployed> oldDeployeds, Set<Deployed> newDeployeds) {
		Set<Deployed> olds = newHashSet(oldDeployeds);
		Set<Deployed> news = newHashSet(newDeployeds);

		for (Deployed old : olds) {
			Deployed aNew = findSimilar(old, news);
			if (aNew != null) {
				news.remove(aNew);
				if (isDifferent(old, aNew)) {
                    builder.modify(old, aNew);
                } else {
                    builder.noOp(aNew);
                }
			} else {
                builder.destroy(old);
            }
		}

		for (Deployed aNew : news) {
            builder.create(aNew);
        }
	}

	@SuppressWarnings("rawtypes")
    static Deployed findSimilar(Deployed old, Set<Deployed> news) {
		for (Deployed aNew : news) {
			if (isSimilar(old, aNew)) {
				return aNew;
			}
		}
		return null;
	}

	@SuppressWarnings("rawtypes")
    static boolean isDifferent(Deployed old, Deployed aNew) {
		if (!old.getType().equals(aNew.getType())) {
			return true;
		}

		Descriptor d = DescriptorRegistry.getDescriptor(old.getType());
		for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
			if (pd.getName().equals(Deployed.DEPLOYABLE_FIELD) || pd.getName().equals(Deployed.CONTAINER_FIELD)) {
				continue;
			}
			if (!pd.areEqual(old, aNew)) {
				return true;
			}
		}
		return isDifferent(old.getDeployable(), aNew.getDeployable());
	}

	static boolean isDifferent(Deployable oldDeployable, Deployable newDeployable) {
		if (!oldDeployable.getType().equals(newDeployable.getType())) {
			return true;
		}

		if (arePropertiesDifferent(oldDeployable, newDeployable)) {
			return true;
		}

		if (oldDeployable instanceof DeployableArtifact) {
			if (isDifferent((DeployableArtifact) oldDeployable, (DeployableArtifact) newDeployable)) {
				return true;
			}
		}

		return false;
	}

	public static boolean arePropertiesDifferent(Deployable oldDeployable, Deployable newDeployable) {
		Descriptor d = DescriptorRegistry.getDescriptor(oldDeployable.getType());
		for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
			if (!pd.areEqual(oldDeployable, newDeployable)) {
				return true;
			}
		}
		return false;
	}

	static boolean isDifferent(DeployableArtifact oldArtifact, DeployableArtifact newArtifact) {
		logger.info("Checking artifact difference for {} and {}", oldArtifact, newArtifact);
		try {
			boolean b = !FileComparisonUtils.contentEquals(((LocalFile) oldArtifact.getFile()).getFile(), ((LocalFile) newArtifact.getFile()).getFile());
			logger.info("Content different? " + b);
			return b;
		} catch (IOException exc) {
			throw new RuntimeException("Cannot compare artifacts " + oldArtifact.getId() + " and " + newArtifact.getId(), exc);
		}
	}

	@SuppressWarnings("rawtypes")
    static boolean isSimilar(Deployed old, Deployed aNew) {
		return aNew.getId().equals(old.getId()) && old.getType().equals(aNew.getType());
	}

	private static final Logger logger = LoggerFactory.getLogger(DeploymentOperationCalculator.class);
}
