package com.xebialabs.deployit.task;

import com.google.common.collect.Lists;
import com.google.common.io.Files;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.EventCallback;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.task.Task.State;
import com.xebialabs.deployit.task.Task.TaskStateChangeEvent;
import javassist.util.proxy.ProxyObjectInputStream;
import javassist.util.proxy.ProxyObjectOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.DirectFieldAccessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.security.core.context.SecurityContextHolder;

import javax.annotation.PostConstruct;
import java.io.*;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.task.Task.State.*;

public class ArchivingTaskRegistry implements TaskRegistry {

	@Autowired
	private TaskArchive taskArchive;

	// needed to be set in the deserialized tasks
	@Autowired
	private RepositoryService repositoryService;

	private File recoveryFile;

	private int recoveryFileWritingIntervalMillis;

	private Registry registry = new Registry();

	private EventCallback<TaskStateChangeEvent> taskArchivingCallback;

	private Timer recoveryFileWritingTimer;

	@Required
	public void setRecoveryFile(File recoveryFile) {
		this.recoveryFile = recoveryFile;
	}

	@Required
	public void setRecoveryFileWritingIntervalMillis(int recoveryFileWritingIntervalMillis) {
		this.recoveryFileWritingIntervalMillis = recoveryFileWritingIntervalMillis;
	}

	@Override
	public String registerTask(final Task task) {
		final String uuid = UUID.randomUUID().toString();
		task.setId(uuid);
		registry.register(task);
		return uuid;
	}

	@Override
	public Task getTask(final String id) {
		return registry.retrieve(id);
	}

	@Override
	public void cancelTask(String id) {
		Task task = getTask(id);
		if (task.isReadyForExecution() || task.getState() == PENDING) {
			boolean moveToArchive = false;
			if (task.getState().equals(STOPPED))
				moveToArchive = true;
			
			task.cancel();//Setting status to CANCELLED before persisting.
			
			if (moveToArchive) {
				moveTaskFromRegistryToArchive(task);
			}
			registry.remove(id);
		}
	}

	@Override
	public Collection<Task> getTasks() {
		return registry.tasks();
	}

	@Override
	public List<Task> getIncompleteTasksForUser(String username) {
		logger.debug("Finding incomplete tasks for user [{}]", username);
		List<Task> myTasks = Lists.newArrayList();
		for (Task eachTask : getAllIncompleteTasks()) {
			if (eachTask.getOwner().equals(username)) {
				myTasks.add(eachTask);
			}
		}
		logger.debug("Returning [{}] tasks", myTasks.size());
		return myTasks;
	}


	@Override
	public List<Task> getAllIncompleteTasks() {
		logger.debug("Finding all incomplete tasks");
		List<Task> myTasks = Lists.newArrayList();
		for (Task eachTask : registry.tasks()) {
			logger.debug("Considering task {}", eachTask);
			
			// although the registry only contains unfinished tasks, there can
			// be a time-window when the task is finished and is yet to be moved
			// to the task archive, so better check the status
			if (eachTask.getState().equals(PENDING) || eachTask.getState().equals(QUEUED) || eachTask.getState().equals(STOPPED) || eachTask.getState().equals(EXECUTING)) {
				logger.debug("Returning task [{}] because it's PENDING, STOPPED or EXECUTING", eachTask);
				myTasks.add(eachTask);
			}
		}
		logger.debug("Returning [{}] tasks", myTasks.size());
		return myTasks;
	}
	
	@Override
	public void assignTask(String taskId, String owner) {
		logger.debug("Assigning task [{}] to owner [{}]", taskId, owner);
		Task task = getTask(taskId);
		task.setOwner(owner);
	}

	@Override
	public void assignMyTask(String taskId, String owner) {
		Task task = getTask(taskId);
		if (task.getOwner().equals(SecurityContextHolder.getContext().getAuthentication().getName())) {
			assignTask(taskId, owner);
		} else {
			throw PermissionDeniedException.withMessage("You are not the current owner of task " + taskId);
		}
	}

	@PostConstruct
	public void afterPropertiesSet() {
		loadTasksFromRecoveryFile();
		registerTaskArchivingCallback();
		startRecoveryFileWritingTimer();
	}

	private void loadTasksFromRecoveryFile() {
		if (recoveryFile != null && recoveryFile.exists()) {
			try {
				ObjectInputStream recoveryIn = new ProxyObjectInputStream(new FileInputStream(recoveryFile));
				try {
					registry.recover(recoveryIn);
					postprocessTaskRegistryAfterRecovery();
				} finally {
					recoveryIn.close();
				}
			} catch (ClassNotFoundException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);
			} catch (IOException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);
			}
		}
	}

	private void postprocessTaskRegistryAfterRecovery() {
		Iterator<Task> tasksIterator = registry.tasks().iterator();
		if (tasksIterator.hasNext()) {
			logger.info("Recovering tasks after server crash");
			while (tasksIterator.hasNext()) {
				Task nextTask = tasksIterator.next();
				State taskState = nextTask.getState();
				if (taskState == PENDING) {
					logger.info("Removing {} task {}", taskState, nextTask.getId());
					tasksIterator.remove();
				} else {
					logger.info("Recovering {} task {}", taskState, nextTask.getId());
					setDepedenciesInTask(nextTask);
					nextTask.processAfterRecovery();
				}
			}
			logger.info("Recovered tasks after server crash");
		}
	}

	private void setDepedenciesInTask(Task eachTask) {
		if (!eachTask.getState().equals(PENDING)) {
			DirectFieldAccessor directFieldAccessor = new DirectFieldAccessor(eachTask);
			if (directFieldAccessor.isWritableProperty("repositoryService"))
				directFieldAccessor.setPropertyValue("repositoryService", repositoryService);
		}
	}

	@Override
	public void destroy() {
		stopRecoveryFileWritingTimer();
        // Make sure that everything is synced up once we're destroyed!
        writeRecoveryFile();
		deregisterTaskArchivingCallback();
	}

	private void registerTaskArchivingCallback() {
		taskArchivingCallback = new EventCallback<TaskStateChangeEvent>() {
			@Override
			public void receive(TaskStateChangeEvent event) {
				Task task = event.getTask();
				if (task.getState().equals(DONE)) {
					moveTaskFromRegistryToArchive(task);
				}
			}

		};
		EventBus.registerForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void deregisterTaskArchivingCallback() {
		EventBus.deregisterForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void startRecoveryFileWritingTimer() {
		recoveryFileWritingTimer = new Timer(ArchivingTaskRegistry.class.getName() + "#recoveryFileWritingTimer", true);
		recoveryFileWritingTimer.schedule(new TimerTask() {
			@Override
			public void run() {
                writeRecoveryFile();
			}
		}, recoveryFileWritingIntervalMillis, recoveryFileWritingIntervalMillis);
	}

	private void stopRecoveryFileWritingTimer() {
		recoveryFileWritingTimer.cancel();
	}

	synchronized void writeRecoveryFile() {
		if (recoveryFile != null) {
			try {
				if (registry.hasInProgressTasks()) {
					logger.debug("Writing task recovery file.");
					File recoveryTemp = new File(recoveryFile.getParentFile(), "recovery.tmp");
					ObjectOutputStream recoveryOut = new ProxyObjectOutputStream(new FileOutputStream(recoveryTemp));
					try {
						registry.writeRecovery(recoveryOut);
					} finally {
						recoveryOut.close();
					}
					deleteRecoveryFile();
					Files.move(recoveryTemp, recoveryFile);
				} else {
					deleteRecoveryFile();
				}
			} catch (IOException exc) {
				logger.error("Cannot write task registry recovery file", exc);
			}
		}
	}

	@SuppressWarnings("deprecation")
    protected void deleteRecoveryFile() {
	    if (recoveryFile.exists()) {
	    	try {
	    		Files.deleteRecursively(recoveryFile);
	    	} catch (IOException e) {
	    		logger.error("Could not delete {}", recoveryFile, e);
	    	}
	    }
    }

	private void moveTaskFromRegistryToArchive(final Task task) {
		checkNotNull(task.getOwnerCredentials(), "Cannot archive task " + task.getId() + " because it has no owner");
		SecurityTemplate.executeAs(task.getOwnerCredentials(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
				if (task instanceof DeploymentTask) {
					taskArchive.archiveTask((DeploymentTask) task);
				}
				registry.remove(task.getId());
				return null;
			}
		});
	}

	public TaskArchive getTaskArchive() {
		return taskArchive;
	}

	private static final Logger logger = LoggerFactory.getLogger(ArchivingTaskRegistry.class);
}
