package com.xebialabs.deployit.io;

import static org.springframework.util.Assert.isTrue;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.InputSupplier;
import com.xebialabs.deployit.util.GuavaFiles;

/**
 */
public class Exploder {

	/**
	 * Extracts a archive input stream.<br/>
	 * <b>N.B.:</b> You should cleanup after yourself, the temp directory won't be automatically deleted after this call.
	 * 
	 * @param inputZip
	 *            the input stream with the archive.
     * @param outputDirectory
     *            the folder where to extract the archive.
	 * @throws IOException
	 *             if an I/O exception occured while extracting.
	 */
	public static void explode(InputSupplier<? extends InputStream> inputZip, File outputDirectory) throws IOException {
		ZipInputStream zipEntryStream = null;
        try {
			zipEntryStream = new ZipInputStream(inputZip.getInput());

			for (;;) {
				ZipEntry entry = zipEntryStream.getNextEntry();
				if (entry == null) {
					return;
				}

				try {
					if (entry.isDirectory()) {
						final File file = new File(outputDirectory, entry.getName());
						if(!file.exists()){
							isTrue(file.mkdirs(), "Could not create directory: " + entry.getName());
						}
						continue;
					}

					final File destinationFile = new File(outputDirectory, entry.getName());
                    final String parentDirPath = destinationFile.getParent();
                    if (parentDirPath != null) {
                        final File destinationDir = new File(parentDirPath);
                        if (!destinationDir.exists()) {
                            isTrue(destinationDir.mkdirs(), "Could not create directory: " + entry.getName());
                        }
					}

                    isTrue(destinationFile.createNewFile(), "Could not create file: " + entry.getName());

					FileOutputStream fo = new FileOutputStream(destinationFile);
					try {
						ByteStreams.copy(zipEntryStream, fo);
					} finally {
						Closeables.closeQuietly(fo);
					}
				} finally {
					zipEntryStream.closeEntry();
				}
			}
		} catch (IOException exc) {
			// On exception, clean up!
			try {
				GuavaFiles.deleteRecursively(outputDirectory);
			} catch (Exception e) {
				logger.error("Could not delete {}", outputDirectory, e);
			}
			throw exc;
		} finally {
	        Closeables.closeQuietly(zipEntryStream);
        }
	}

	private static final Logger logger = LoggerFactory.getLogger(Exploder.class);

}
