package com.xebialabs.deployit.service.deployment;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner.hasPlaceholders;
import static java.lang.String.format;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;

@Component
public class DeployedGenerator {

	private RepositoryService repository;
	private TypeCalculator calculator;

	@Autowired
	public DeployedGenerator(RepositoryService repository, TypeCalculator calculator) {
		this.calculator = calculator;
		this.repository = repository;
	}



    public Deployed<?, ?> createMostSpecificDeployed(final Deployable deployable, final Container container, final ConsolidatedDictionary dictionary) {
	    return generateMostSpecificDeployed(deployable, container, dictionary, true);
    }

	public Deployed<?, ?> generateMostSpecificDeployed(final Deployable deployable, final Container container, final ConsolidatedDictionary dictionary) {
		return generateMostSpecificDeployed(deployable, container, dictionary, false);
	}

	private Deployed<?, ?> generateMostSpecificDeployed(final Deployable deployable, final Container container, final ConsolidatedDictionary dictionary, boolean manualAction) {
        Type deployedType = calculator.findMostSpecificDeployedTypeForDeployableAndContainerTypes(deployable.getType(), container.getType());
        if (deployedType != null) {
	        // If it was a manual action (as opposed to a automatic generation), ignore the tags.
	        if (manualAction || dontUseTagsOrTagsMatch(deployable, container)) {
                return generateDeployed(deployable, container, deployedType, dictionary);
	        }
        }
        return null;
    }

	private boolean dontUseTagsOrTagsMatch(Deployable deployable, Container container) {
		Set<String> deployableTags = emptySetIfNull(deployable.getTags());
		Set<String> containerTags = emptySetIfNull(container.getTags());
		logger.debug("Using tags: deployable [{}] and container [{}]", deployableTags, containerTags);
		return (deployableTags.isEmpty() && containerTags.isEmpty()) || !Sets.intersection(deployableTags, containerTags).isEmpty();
	}

    private Set<String> emptySetIfNull(Set<String> tags) {
        if (tags == null) {
            return newHashSet();
        }
        return tags;
    }

    public Deployed<?, ?> generateDeployed(final Deployable deployable, final Container container, final Type deployedType, final ConsolidatedDictionary dictionary) {
        Deployed<Deployable, Container> deployed = basicDeployed(deployable, container, deployedType);
        setPropertyPlaceholders(deployed, deployable, dictionary);
		setFilePlaceholders(deployed, deployable, dictionary);
        return deployed;
    }

	private Deployed<Deployable, Container> basicDeployed(final Deployable deployable, final Container container, final Type deployedType) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployedType);

	    Deployed<Deployable, Container> deployed = deployedDesc.newInstance();
	    deployed.setDeployable(deployable);
        deployed.setContainer(container);
        IdGenerator.generateId(deployed);

        return deployed;
    }

	private void setPropertyPlaceholders(Deployed<Deployable, Container> deployed, final Deployable deployable, final ConsolidatedDictionary dictionary) {
        Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployed.getType());
	    Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployable.getType());

	    for (PropertyDescriptor deployedPropDesc : deployedDesc.getPropertyDescriptors()) {
	        String name = deployedPropDesc.getName();
	        if (name.equals("placeholders")) {
	        	continue;
	        }

	        PropertyDescriptor deployablePropDesc = deployableDesc.getPropertyDescriptor(name);
		    if (deployablePropDesc != null) {
			    Object value = deployablePropDesc.get(deployable);
			    String stringValue = (value == null) ? null : value.toString();
			    if (Strings.isNullOrEmpty(stringValue)) {
				    setDeployedFromDictionary(deployed, dictionary, deployedPropDesc, deployablePropDesc);
			    } else {
					try {
						value = dictionary.resolve(value, deployablePropDesc);
						value = convertToReferenceIfNeeded(value, deployablePropDesc, deployedPropDesc);
						deployedPropDesc.set(deployed, value);
					} catch (DictionaryValueException exc) {
						logger.warn("Could not resolve dictionary keys for property " + deployablePropDesc + " on " + deployed + ". Property will be left empty.");
						logger.trace("Exception was", exc);
					} catch (RuntimeException exc) {
						logger.warn("Could not convert (resolved) value to correct type for property " + deployablePropDesc + " on " + deployed + ". Property will be left empty.");
						logger.trace("Exception was", exc);
					}
			    }
		    } else {
			    setDeployedFromDictionary(deployed, dictionary, deployedPropDesc, null);
		    }
        }
    }

	private void setDeployedFromDictionary(Deployed<Deployable, Container> deployed, ConsolidatedDictionary dictionary, PropertyDescriptor deployedPropDesc, PropertyDescriptor deployablePropDesc) {
		if (dictionary.containsKey(deployedPropDesc.getFqn())) {
			Object value = dictionary.get(deployedPropDesc.getFqn());
			value = convertToReferenceIfNeeded(value, deployablePropDesc, deployedPropDesc);
			deployedPropDesc.set(deployed, value);
		}
	}

	private void setFilePlaceholders(final Deployed<Deployable, Container> deployed, final Deployable deployable, final ConsolidatedDictionary dictionary) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployed.getType());
		Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployable.getType());

		PropertyDescriptor deployedPlaceholdersDesc = deployedDesc.getPropertyDescriptor("placeholders");
		if (deployedPlaceholdersDesc != null) {
	        PropertyDescriptor deployableplaceholders = deployableDesc.getPropertyDescriptor("placeholders");
	        @SuppressWarnings("unchecked")
            Set<String> placeholders = (Set<String>) deployableplaceholders.get(deployable);

	        deployedPlaceholdersDesc.set(deployed, resolveFilePlaceholders(Maps.<String, String>newHashMap(), placeholders, dictionary));
		}
    }

	public Deployed<? ,?> generateUpgradedDeployed(final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
        Deployed<?, ?> upgradedDeployed = createUpgradedDeployed(newDeployable, existingDeployed);
        setUpgradedPropertyPlaceholders(upgradedDeployed, newDeployable, existingDeployed, dictionary);
		setUpgradedFilePlaceholders(upgradedDeployed, newDeployable, existingDeployed, dictionary);
        return upgradedDeployed;
    }

	private Deployed<?, ?> createUpgradedDeployed(final Deployable newDeployable, final Deployed<?, ?> existingDeployed) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(existingDeployed.getType());
        Type deployableType = deployedDesc.getDeployableType();
        Type newDeployableType = newDeployable.getType();
		checkArgument(deployableType.equals(newDeployableType) || DescriptorRegistry.getSubtypes(deployableType).contains(newDeployableType),
		        "The new Deployable type %s should be assignable to %s", newDeployableType, deployableType);

        Deployed<Deployable, Container> upgradedDeployed = deployedDesc.newInstance();
        upgradedDeployed.setDeployable(newDeployable);
        upgradedDeployed.setId(existingDeployed.getId());

        return upgradedDeployed;
    }

	private void setUpgradedPropertyPlaceholders(final Deployed<?, ?> upgradedDeployed, final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
        Descriptor deployedDesc = DescriptorRegistry.getDescriptor(existingDeployed.getType());
        Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployedDesc.getDeployableType());

        for (PropertyDescriptor deployedPropDesc : deployedDesc.getPropertyDescriptors()) {
            String name = deployedPropDesc.getName();
	        if (name.equals("placeholders") || name.equals("deployable")) {
		        continue;
	        }

	        Object oldDeployedValue = deployedPropDesc.get(existingDeployed);
            Object valueToSet = oldDeployedValue;

            PropertyDescriptor deployablePropDesc = deployableDesc.getPropertyDescriptor(name);
            if (deployablePropDesc != null) {
	            Object oldDeployableValue = deployablePropDesc.get(existingDeployed.getDeployable());
	            String oldDeployedValueAsString = oldDeployedValue != null ? oldDeployedValue.toString() : null;
                if ((oldDeployableValue != null) && (oldDeployableValue.equals(oldDeployedValueAsString) || hasPlaceholders(oldDeployableValue.toString()))) {
                    valueToSet = deployablePropDesc.get(newDeployable);
	                try {
		                valueToSet = dictionary.resolve(valueToSet, deployablePropDesc);
		                valueToSet = convertToReferenceIfNeeded(valueToSet, deployablePropDesc, deployedPropDesc);
	                } catch (DictionaryValueException exc) {
		                valueToSet = oldDeployedValue;
		                logger.warn("Could not resolve dictionary keys for property " + deployedPropDesc + " of " + upgradedDeployed + ". Property will be set to old value.");
		                logger.trace("Exception was: ", exc);
	                }
                }
            }

            try {
                deployedPropDesc.set(upgradedDeployed, valueToSet);
	        } catch (RuntimeException exc) {
		        logger.error("Could not convert (resolved) value to correct type for property " + deployedPropDesc + " of " + upgradedDeployed + ". Property will be left empty.", exc);
	        }
        }
    }

    private Object convertToReferenceIfNeeded(Object valueToSet, PropertyDescriptor deployablePropDesc, PropertyDescriptor deployedPropDesc) {
		if (valueToSet == null) return valueToSet;
	    if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.CI) {
			return repository.read(valueToSet.toString());
		} else if (optionalKind(deployablePropDesc, PropertyKind.SET_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.SET_OF_CI) {
			return Sets.<ConfigurationItem>newHashSet(getCollectionOfCis(valueToSet));
		} else if (optionalKind(deployablePropDesc,PropertyKind.LIST_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI) {
			return Lists.<ConfigurationItem>newArrayList(getCollectionOfCis(valueToSet));
		} 
		return valueToSet;
	}

	private boolean optionalKind(PropertyDescriptor deployablePropDesc, PropertyKind kind) {
		return deployablePropDesc == null || deployablePropDesc.getKind() == kind;
	}

	@SuppressWarnings("unchecked")
    private Collection<ConfigurationItem> getCollectionOfCis(Object valueToSet) {
	    Collection<ConfigurationItem> cis = newArrayList();
	    for (String s : (Collection<String>) valueToSet) {
	    	cis.add(repository.read(s));
	    }
	    return cis;
    }

	private void setUpgradedFilePlaceholders(final Deployed<?, ?> upgradedDeployed, final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
		Descriptor deployedDesc = DescriptorRegistry.getDescriptor(upgradedDeployed.getType());
		Descriptor deployableDesc = DescriptorRegistry.getDescriptor(newDeployable.getType());

		PropertyDescriptor deployedPlaceholdersDesc = deployedDesc.getPropertyDescriptor("placeholders");
		if (deployedPlaceholdersDesc != null) {
			@SuppressWarnings("unchecked")
            Map<String, String> existingDeployedPlaceholders = (Map<String, String>) deployedPlaceholdersDesc.get(existingDeployed);
			@SuppressWarnings("unchecked")
            Set<String> newDeployablePlaceholders = (Set<String>) deployableDesc.getPropertyDescriptor("placeholders").get(newDeployable);

			deployedPlaceholdersDesc.set(upgradedDeployed, resolveFilePlaceholders(existingDeployedPlaceholders, newDeployablePlaceholders, dictionary));
		}
	}

	private Map<String, String> resolveFilePlaceholders(final Map<String, String> existingDeployedPlaceholders, final Set<String> deployablePlaceholders, final ConsolidatedDictionary dictionary) {
		Map<String, String> newDeployedPlaceholders = newHashMap();
		for (String deployablePlaceholder : deployablePlaceholders) {
			String placeholderValue;
			try {
				placeholderValue = (String) dictionary.resolve("{{" + deployablePlaceholder + "}}");
			} catch (DictionaryValueException ignored) {
				if (existingDeployedPlaceholders.containsKey(deployablePlaceholder)) {
					placeholderValue = existingDeployedPlaceholders.get(deployablePlaceholder);
				} else {
					placeholderValue = "";
				}
			}
			newDeployedPlaceholders.put(deployablePlaceholder, placeholderValue);
		}
		return newDeployedPlaceholders;
	}

	private static final Logger logger = LoggerFactory.getLogger(DeployedGenerator.class);
}
