package com.xebialabs.deployit.service.replacement;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Dictionary;
import com.xebialabs.deployit.service.deployment.DictionaryValueException;

public class ConsolidatedDictionary {

	private Map<String, String> entries = newHashMap();

	public static ConsolidatedDictionary create(Collection<Dictionary> dicts) {
		ConsolidatedDictionary dictionary = new ConsolidatedDictionary();
		if (dicts != null) {
			for (Dictionary dict : dicts) {
				dictionary.addDictionary(dict);
			}
		}

		return dictionary;
	}

	void addDictionary(Dictionary d) {
		for (String k : d.getEntries().keySet()) {
			if (entries.containsKey(k) && !entries.get(k).equals(d.getEntries().get(k))) {
				throw new IllegalArgumentException("Flattened Dictionary already contains a value for key " + k);
			}
			entries.put(k, d.getEntries().get(k));
		}
	}

	public Map<String, String> getEntries() {
		return entries;
	}

	public boolean containsKey(String key) {
		return entries.containsKey(key);
	}

	public String get(String key) {
		return entries.get(key);
	}

	public String resolve(String value) throws DictionaryValueException {
		MustachePlaceholderReplacer placeholderReplacer = new MustachePlaceholderReplacer(entries, false);
		if (value != null) {
			value = placeholderReplacer.replace(value);
		}
		return value;
	}

	@SuppressWarnings("unchecked")
    public Object resolve(Object value, PropertyDescriptor deployablePropDesc) throws DictionaryValueException {
		if (value == null) return null;
		MustachePlaceholderReplacer placeholderReplacer = new MustachePlaceholderReplacer(entries, false);
		switch (deployablePropDesc.getKind()) {
			case BOOLEAN:
			case INTEGER:
			case SET_OF_CI:
			case LIST_OF_CI:
			case CI:
			case ENUM:
				return value;
			case STRING:
				return placeholderReplacer.replace((String) value);
			case SET_OF_STRING:
				return Sets.<String>newHashSet(resolveCollection((Set<String>) value, placeholderReplacer));
			case LIST_OF_STRING:
				return Lists.<String>newArrayList(resolveCollection((List<String>) value, placeholderReplacer));
			case MAP_STRING_STRING:
				return resolveMap((Map<String, String>) value, placeholderReplacer);
			default:
				throw new IllegalStateException("Unknown property kind: " + deployablePropDesc.getKind());
		}
	}

	private Object resolveMap(Map<String, String> value, MustachePlaceholderReplacer placeholderReplacer) {
		Map<String, String> resolvedMap = newHashMap();
		for (Map.Entry<String, String> entry : value.entrySet()) {
			try {
				resolvedMap.put(entry.getKey(), placeholderReplacer.replace(entry.getValue()));
			} catch (DictionaryValueException e) {
				resolvedMap.put(entry.getKey(), entry.getValue());
			}
		}
		return resolvedMap;
	}

	private Collection<String> resolveCollection(Collection<String> value, MustachePlaceholderReplacer placeholderReplacer) {
		Collection<String> resolved = newArrayList();
		for (String s : value) {
			try {
				resolved.add(placeholderReplacer.replace(s));
			} catch (DictionaryValueException e) {
				resolved.add(s);
			}
		}
		return resolved;
	}
}
