package com.xebialabs.deployit.task.deployment;

import static com.google.common.collect.Lists.newArrayList;

import java.util.List;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.repository.ChangeSet;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.task.DeploymentTask;
import org.springframework.security.core.Authentication;

@SuppressWarnings("serial")
public class UndeploymentTask extends DeploymentTask {

    private DeltaSpecification specification;
    private DeployedApplication existingDeployment;

    public UndeploymentTask(DeltaSpecification specification, final DeployedApplication existingDeployment, final List<? extends DeploymentStep> steps, final RepositoryService repositoryService, final WorkDir workDir) {
		super(steps, repositoryService, workDir);
        this.specification = specification;
        this.existingDeployment = existingDeployment;
        initFromDeployment(existingDeployment);
    }

    @Override
    public void performPreFlightChecks() {
        Authentication owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner, new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
                ChangeSet changeset = determineRepositoryChanges();
                repositoryService.checkReferentialIntegrity(changeset);
				return null;
			}
		});
    }

    private ChangeSet determineRepositoryChanges() {
        List<String> roesToDelete = newArrayList();
        // First delete the deployment, otherwise, the mappings cannot be deleted as they're referenced.
        roesToDelete.add(existingDeployment.getId());
        for (Delta delta : specification.getDeltas()) {
            // All delta's should be DESTROY...
            roesToDelete.add(delta.getPrevious().getId());
        }

        ChangeSet batchUpdate = new ChangeSet();
        batchUpdate.setDeleteCiIds(roesToDelete);
        return batchUpdate;
    }

    @Override
	public void doAfterTaskStateChangedToDone() {
		Authentication owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner, new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
                ChangeSet changeset = determineRepositoryChanges();
                repositoryService.execute(changeset);
				return null;
			}
		});
	}

    @Override
    public DeploymentType getDeploymentType() {
        return DeploymentTask.DeploymentType.UNDEPLOY;
    }
}
