package com.xebialabs.deployit.upgrade;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Multimaps.index;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.ReflectionsHolder.getSubTypesOf;
import static com.xebialabs.deployit.server.api.upgrade.Version.VERSION_0;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.google.common.collect.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.boot.PluginVersions;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;

@Component
public class Upgrader {

    private ApplicationContext context;

    private JcrTemplate jcrTemplate;
    
    private ListMultimap<String, Upgrade> upgradeBeans;

	private boolean questionAsked = false;
	
    public @Autowired Upgrader(ApplicationContext context, JcrTemplate jcrTemplate) {
        this.context = context;
        this.jcrTemplate = jcrTemplate;
    }

    @PostConstruct
    public void doUpgrade() {
        upgradeBeans = getUpgrades();
	    Set<String> registeredPlugins = PluginVersions.getRegisteredPlugins();
	    for (String component : Iterables.concat(newArrayList("deployit"), registeredPlugins)) {
		    logger.debug("Checking component [{}] for upgrades", component);
		    upgradeComponent(component);
	    }
    }


    void upgradeComponent(String component) {
        final Version componentVersion = readVersionOfComponent(component);
        List<Upgrade> applicableUpgrades = filterUpgrades(ensureNotNull(upgradeBeans.get(component)), componentVersion);
        if (!applicableUpgrades.isEmpty()) {
	        Collections.sort(applicableUpgrades);
            logger.info("Checked component [{}] which is at version [{}] -> Found upgrades to run: {}", new Object[] {component, componentVersion, applicableUpgrades});
            if (!questionAsked) {
		        System.out.println("*** WARNING ***");
		        System.out.println("We detected that we need to upgrade your Deployit repository");
		        System.out.println("Before continuing we suggest you backup your Deployit repository in case the upgrade fails.");
		        System.out.println("Please ensure you have 'INFO' level logging configured.");
		        System.out.print("Please enter 'yes' if you want to continue [no]: ");
		        String response = read();
		        if (!response.equalsIgnoreCase("yes")) {
			        EventBus.publish(new ShutdownEvent());
			        return;
		        }
		        questionAsked = true;
	        }
            applyUpgrades(applicableUpgrades);
        } else if (componentVersion.equals(Version.valueOf(component, VERSION_0))) {
	        Version currentPluginVersion = Version.valueOf(component, PluginVersions.getVersionFor(component));
	        logger.info("Registering previous unregistered version: [{}]", currentPluginVersion);
	        storeVersionOfComponent(currentPluginVersion);
        }
    }

    void storeVersionOfComponent(final Version version) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws IOException, RepositoryException {
                Node node = getVersionsNode(session);
                node.setProperty(version.getComponent(), version.getVersion());
                session.save();
                return null;
            }
        });
    }

    Version readVersionOfComponent(final String component) {
        return jcrTemplate.execute(new JcrCallback<Version>() {
            @Override
            public Version doInJcr(Session session) throws IOException, RepositoryException {
                Node node = getVersionsNode(session);
                if (node.hasProperty(component)) {
                    Property versionProp = node.getProperty(component);
                    return Version.valueOf(component, versionProp.getString());
                } else {
                    return Version.valueOf(component, VERSION_0);
                }
            }
        });
    }

    private Node getVersionsNode(Session session) throws RepositoryException {
        try {
            return session.getNode(VERSIONS_NODE_ID);
        } catch (PathNotFoundException pnfe) {
            return session.getRootNode().addNode(VERSIONS_NODE_NAME);
        }
    }

    private void applyUpgrades(final List<Upgrade> applicableUpgrades) {
	    final ImmutableListMultimap<Version,Upgrade> versionToUpgradeMap = Multimaps.index(applicableUpgrades, new Function<Upgrade, Version>() {
		    public Version apply(Upgrade input) {
			    return input.upgradeVersion();
		    }
	    });

	    List<Version> versions = newArrayList(Lists.transform(applicableUpgrades, new Function<Upgrade, Version>() {
		    public Version apply(Upgrade input) {
			    return input.upgradeVersion();
		    }
	    }));

	    Collections.sort(versions);

	    for (final Version version : versions) {
		    logger.info("Upgrading to version [{}]", version);
		    jcrTemplate.execute(new JcrCallback<Object>() {
			    @Override
			    public Object doInJcr(Session session) throws IOException, RepositoryException {
				    for (Upgrade applicableUpgrade : versionToUpgradeMap.get(version)) {
					    if (!applicableUpgrade.doUpgrade(new RawRepositoryImpl(session))) {
						    throw new UpgradeException("Could not perform upgrade %s to upgrade to %s", applicableUpgrade.getClass(), applicableUpgrade.upgradeVersion());
					    }
				    }

				    // Do a per-component transaction.
				    session.save();
				    return null;
			    }
		    });
		    storeVersionOfComponent(version);
	    }
    }

    private List<Upgrade> filterUpgrades(Collection<Upgrade> upgradeBeans, final Version repoVersion) {
        List<Upgrade> applicableUpgrades = newArrayList(filter(upgradeBeans, new Predicate<Upgrade>() {
            @Override
            public boolean apply(Upgrade input) {
                return input.shouldBeApplied(repoVersion);
            }
        }));

        Collections.sort(applicableUpgrades);
        return applicableUpgrades;
    }

    protected ListMultimap<String, Upgrade> getUpgrades() {
        Set<Class<? extends Upgrade>> upgradeClasses = getSubTypesOf(Upgrade.class);
        return index(transform(upgradeClasses, new Function<Class<? extends Upgrade>, Upgrade>() {
            @Override
            public Upgrade apply(Class<? extends Upgrade> input) {
                return context.getAutowireCapableBeanFactory().createBean(input);
            }
        }), new Function<Upgrade, String>() {
            @Override
            public String apply(Upgrade input) {
                return input.upgradeVersion().getComponent();
            }
        });
    }

	protected String read() {
		BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
		try {
			final String line = stdin.readLine();
			if (line != null) {
				return line.trim();
			} else
				return line;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	private static <T> List<T> ensureNotNull(List<T> list) {
		return list == null ? Lists.<T>newArrayList() : list;
	}
	
	private static final Logger logger = LoggerFactory.getLogger(Upgrader.class);
}
