/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import java.io.File;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.conversion.StringCoder;
import com.xebialabs.deployit.util.DeployitKeyStore;
import com.xebialabs.deployit.util.PasswordEncrypter;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;

public class DeployitConfig {

    // ******************
    // Spring Defaults
    // ******************
    public static final String SPRING_CTX_CFG_LOCATION_KEY = "contextConfigLocation";
    public static final String SPRING_CONFIG = "classpath:spring/deployit-context.xml,classpath:deployit-security.xml,classpath*:hotfix-context.xml";

    // ******************
    // Jetty HTTP Server Defaults
    // ******************
    public static final String DEFAULT_WEBCONTENT_PACKAGE = "web";
    public static final String DEFAULT_WEB_CONTEXT_ROOT = "/";
    public static final String DEFAULT_ADMIN_PASSWORD = "admin";
    public static final String DEFAULT_DEPLOYIT_KEYS_KEYSTORE_PASSWORD = "deployit";
    public static final int DEFAULT_HTTP_PORT = 4516;
    public static final int DEFAULT_HTTPS_PORT = 4517;
    public static final int DEFAULT_MAX_THREADS = 24;
    public static final int DEFAULT_MIN_THREADS = 3;
    public static final String FILE_URL_PREFIX = "file://";
    public static final String DEFAULT_JCR_REPOSITORY_PATH = "repository";
    public static final String DEFAULT_IMPORTABLE_PACKAGES_PATH = "importablePackages";
    public static final String DEFAULT_HTTP_BIND_ADDRESS = "0.0.0.0";

    // ******************
    // Property file keys
    // ******************
    public static final String KEY_USE_SSL = "ssl";
    public static final String KEY_SSL_PROTOCOL = "ssl.protocol";
    public static final String KEY_KEYSTORE_PATH = "keystore.path";
    public static final String KEY_KEYSTORE_PASSWORD = "keystore.password";
    public static final String KEY_KEYSTORE_KEYPASSWORD = "keystore.keypassword";
    public static final String KEY_DEPLOYIT_KEYS_KEYSTORE_PASSWORD = "deployit-keys.keystore.password";
    public static final String KEY_HTTP_BIND_ADDRESS = "http.bind.address";
    public static final String KEY_HTTP_PORT = "http.port";
    public static final String KEY_HTTP_CONTEXT_ROOT = "http.context.root";
    public static final String KEY_JCR_REPOSITORY_PATH = "jcr.repository.path";
    public static final String KEY_IMPORTABLE_PACKAGES_PATH = "importable.packages.path";
    public static final String KEY_MINIMUM_THREADS = "threads.min";
    public static final String KEY_MAXIMUM_THREADS = "threads.max";
    public static final String KEY_DEFAULT_PREFIX = "default.";

    public static final String KEY_ADMIN_PASSWORD = "admin.password";

    public static final String KEY_CUSTOM_PREFIX = "custom.";

    // ******************
    // Mandatory
    // ******************
    protected String keyStorePath;
    protected String keyStorePassword;
    protected String keyStoreKeyPassword;
    private String httpBindAddress;
    protected int httpPort;
    private String webContextRoot = DEFAULT_WEB_CONTEXT_ROOT;
    protected String jcrRepositoryPath;
    protected String importablePackagesPath;
    protected boolean ssl;
    protected String sslProtocol;
    protected int minThreads;
    protected String adminPassword;
    private boolean dirty;
    protected int maxThreads;

    private boolean newConfiguration;

    public DeployitConfig() {
        this.newConfiguration = true;
    }

    public void load(Properties properties) {
        ssl = "true".equalsIgnoreCase(properties.getProperty(KEY_USE_SSL, "false"));
        sslProtocol = properties.getProperty(KEY_SSL_PROTOCOL);
        keyStorePath = properties.getProperty(KEY_KEYSTORE_PATH);
        keyStorePassword = properties.getProperty(KEY_KEYSTORE_PASSWORD);
        keyStoreKeyPassword = properties.getProperty(KEY_KEYSTORE_KEYPASSWORD);

        httpBindAddress = properties.getProperty(KEY_HTTP_BIND_ADDRESS, httpBindAddress);
        httpPort = parseIntSilent(properties, KEY_HTTP_PORT, httpPort);
        webContextRoot = properties.getProperty(KEY_HTTP_CONTEXT_ROOT, webContextRoot);
        minThreads = parseIntSilent(properties, KEY_MINIMUM_THREADS, minThreads);
        maxThreads = parseIntSilent(properties, KEY_MAXIMUM_THREADS, maxThreads);

        jcrRepositoryPath = properties.getProperty(KEY_JCR_REPOSITORY_PATH, jcrRepositoryPath);
        importablePackagesPath = properties.getProperty(KEY_IMPORTABLE_PACKAGES_PATH, importablePackagesPath);

        adminPassword = properties.getProperty(KEY_ADMIN_PASSWORD, DEFAULT_ADMIN_PASSWORD);

        if (PasswordEncrypter.getInstance().isEncrypted(adminPassword)) {
            adminPassword = PasswordEncrypter.getInstance().decrypt(adminPassword);
        } else {
            logger.info("Read plaintext password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }

        this.newConfiguration = false;
    }

    private static int parseIntSilent(Properties properties, String KEY, int defaultValue) {
        try {
            return Integer.parseInt(properties.getProperty(KEY));
        } catch (NumberFormatException nfe) {
            return defaultValue;
        }
    }

    public void save(Properties properties) {
        properties.setProperty(KEY_USE_SSL, Boolean.toString(isSsl()));
        if (isSsl()) {
            properties.setProperty(KEY_KEYSTORE_PATH, getKeyStorePath());
            properties.setProperty(KEY_KEYSTORE_PASSWORD, this.keyStorePassword);
            properties.setProperty(KEY_KEYSTORE_KEYPASSWORD, this.keyStoreKeyPassword);
            if (getSslProtocol() != null) {
                properties.setProperty(KEY_SSL_PROTOCOL, getSslProtocol());
            }
        }

        properties.setProperty(KEY_HTTP_BIND_ADDRESS, httpBindAddress);
        properties.setProperty(KEY_HTTP_PORT, Integer.toString(getHttpPort()));
        properties.setProperty(KEY_HTTP_CONTEXT_ROOT, webContextRoot);
        properties.setProperty(KEY_JCR_REPOSITORY_PATH, getJcrRepositoryPath());
        properties.setProperty(KEY_IMPORTABLE_PACKAGES_PATH, getImportablePackagesPath());

        properties.setProperty(KEY_MINIMUM_THREADS, Integer.toString(minThreads));
        properties.setProperty(KEY_MAXIMUM_THREADS, Integer.toString(maxThreads));
        properties.setProperty(KEY_ADMIN_PASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(adminPassword != null ? adminPassword : DEFAULT_ADMIN_PASSWORD));
    }

    public boolean isValid() {
        boolean valid = true;
        if (isSsl()) {
            if (isBlank(getKeyStorePath())) {
                System.err.println(KEY_KEYSTORE_PATH + " must be set.");
                valid = false;
            } else if (!new File(getKeyStorePath()).exists()) {
                System.err.println(KEY_KEYSTORE_PATH + " must point to an existing keystore.");
                valid = false;
            }

        }

        valid &= checkSet(getHttpBindAddress(), KEY_HTTP_BIND_ADDRESS);
        valid &= validateThatIntIsBiggerThen(KEY_HTTP_PORT, getHttpPort(), 0);
        valid &= checkSet(getWebContextRoot(), KEY_HTTP_CONTEXT_ROOT);
        if (!getWebContextRoot().startsWith("/")) {
            System.err.println(KEY_HTTP_CONTEXT_ROOT + " should start with a '/'");
            valid = false;
        }
        valid &= checkSet(getJcrRepositoryPath(), KEY_JCR_REPOSITORY_PATH);
        valid &= checkSet(getImportablePackagesPath(), KEY_IMPORTABLE_PACKAGES_PATH);
        valid &= validateThatIntIsBiggerThen(KEY_MINIMUM_THREADS, getMinThreads(), 0);
        valid &= validateThatIntIsBiggerThen(KEY_MAXIMUM_THREADS, getMaxThreads(), getMinThreads());

        return valid;
    }

    private static boolean checkSet(String value, String key) {
        if (isBlank(value)) {
            System.err.println(key + " must be set.");
            return false;
        }
        return true;
    }

    private static boolean validateThatIntIsBiggerThen(String key, int intToValidate, int shouldBeBiggerThen) {
        if (intToValidate <= shouldBeBiggerThen) {
            System.err.println(key + " is not valid, must be a number larger then " + shouldBeBiggerThen);
            return false;
        }
        return true;
    }

    public boolean isNewConfiguration() {
        return newConfiguration;
    }

    public void setSsl(boolean ssl) {
        this.ssl = ssl;
    }

    public boolean isSsl() {
        return ssl;
    }


    public String getKeyStorePath() {
        return keyStorePath;
    }

    public void setKeyStorePath(String keyStorePath) {
        this.keyStorePath = keyStorePath;
    }

    public String getKeyStorePassword() {
        return StringCoder.decode(keyStorePassword);
    }

    public void setKeyStorePassword(String keyStorePassword) {
        this.keyStorePassword = StringCoder.encode(keyStorePassword);
    }

    public String getKeyStoreKeyPassword() {
        return StringCoder.decode(keyStoreKeyPassword);
    }

    public void setKeyStoreKeyPassword(String keyStoreKeyPassword) {
        this.keyStoreKeyPassword = StringCoder.encode(keyStoreKeyPassword);
    }

    public String getHttpBindAddress() {
        return httpBindAddress;
    }

    public void setHttpBindAddress(String httpBindAddress) {
        this.httpBindAddress = httpBindAddress;
    }

    public int getHttpPort() {
        return httpPort;
    }

    public void setHttpPort(int httpPort) {
        this.httpPort = httpPort;
    }

    public String getJcrRepositoryPath() {
        return jcrRepositoryPath;
    }


    public void setJcrRepositoryPath(final String jcrRepositoryPath) {
        this.jcrRepositoryPath = jcrRepositoryPath;
    }

    public String getImportablePackagesPath() {
        return importablePackagesPath;
    }

    public void setImportablePackagesPath(String setImportablePackagesPath) {
        this.importablePackagesPath = setImportablePackagesPath;
    }

    public String getSslProtocol() {
        return sslProtocol;
    }

    public void setSslProtocol(String sslProtocol) {
        this.sslProtocol = sslProtocol;
    }

    public int getMinThreads() {
        return minThreads;
    }

    public void setMinThreads(int minThreads) {
        this.minThreads = minThreads;
    }

    public int getMaxThreads() {
        return maxThreads;
    }

    public void setMaxThreads(int maxThreads) {
        this.maxThreads = maxThreads;
    }

    public boolean isDirty() {
        return dirty;
    }

    public String getAdminPassword() {
        return adminPassword;
    }

    public void setAdminPassword(String adminPassword) {
        this.adminPassword = adminPassword;
    }

    public String getWebContextRoot() {
        return webContextRoot;
    }

    public void setWebContextRoot(String webContextRoot) {
        this.webContextRoot = webContextRoot;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitConfig.class);
}