package com.xebialabs.deployit;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Closeables;
import com.google.common.io.Files;

public final class DeployitConfiguration {

    public static final String CONFIGURATION_FILENAME = "deployit.conf";
    public static final File DEFAULT_CONFIGURATION_FILE = new File("conf", CONFIGURATION_FILENAME);

    private static final AtomicReference<DeployitConfig> CONFIG = new AtomicReference<DeployitConfig>();

    public static void load() {
        load(DEFAULT_CONFIGURATION_FILE);
    }

    public static void load(File file) {
        logger.debug("Loading Deployit configuration from {}", file.getAbsolutePath());
        CONFIG.compareAndSet(null, loadConfig(file));
    }

    public static boolean exists() {
        return DEFAULT_CONFIGURATION_FILE.exists();
    }

    public static DeployitConfig getInstance() {
        return CONFIG.get();
    }

    public static void writeConfiguration(DeployitConfig deployitConfig) {
        writeConfiguration(DEFAULT_CONFIGURATION_FILE, deployitConfig);
    }

    public static void readIntoConfiguration(DeployitConfig deployitConfig) {
        deployitConfig.load(readConfigurationFile(DEFAULT_CONFIGURATION_FILE));
    }

    static DeployitConfig loadConfig(File configFile) {
        if (configFile.exists()) {
            DeployitConfig deployitConfig = readConfiguration(configFile);
            if(!deployitConfig.isValid()) {
                throw new Error("Configuration " + configFile + " is not valid please run with -setup option.");
            }

            if (deployitConfig.isDirty()) {
                logger.info("Saving (dirty) configuration");
                writeConfiguration(configFile, deployitConfig);
            }

            return deployitConfig;
        } else {
            throw new Error("Configuration " + configFile
                    + " does not exist, please run with -setup option before starting the server or command line interface.");
        }
    }

    static DeployitConfig readConfiguration(File configFile) {
        DeployitConfig deployitConfig = new DeployitConfig();
        deployitConfig.load(readConfigurationFile(configFile));
        return deployitConfig;
    }

    static Properties readConfigurationFile(File configFile) {
        Properties properties = new Properties();
        FileInputStream inStream = null;
        try {
            inStream = Files.newInputStreamSupplier(configFile).getInput();
            properties.load(inStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            Closeables.closeQuietly(inStream);
        }

        return properties;
    }

    static void writeConfiguration(File configFile, DeployitConfig deployitConfig) {
        Properties properties = new Properties();
        deployitConfig.save(properties);
        FileOutputStream output = null;
        try {
            output = Files.newOutputStreamSupplier(configFile).getOutput();
            properties.store(output, "Deployit configuration file.");
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            Closeables.closeQuietly(output);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitConfiguration.class);
}
