package com.xebialabs.deployit.core;

import static java.lang.reflect.Array.newInstance;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

import org.apache.commons.lang.builder.EqualsBuilder;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterators;

@SuppressWarnings("serial")
public class AbstractStringView<T extends Collection<StringValue>> implements Collection<String>, Serializable {
    protected final T wrapped;

    protected AbstractStringView(T wrapped) {
        this.wrapped = wrapped;
    }

    @Override
    public int size() {
        return wrapped.size();
    }

    @Override
    public boolean isEmpty() {
        return wrapped.isEmpty();
    }

    @Override
    public boolean contains(Object o) {
        return wrapped.contains(stringToStringValue(o));
    }

    @Override
    public Iterator<String> iterator() {
        return Iterators.transform(wrapped.iterator(), new Function<StringValue, String>() {
            @Override
            public String apply(StringValue input) {
                return input.toString();
            }
        });
    }

    @Override
    public Object[] toArray() {
        Object[] objects = new Object[size()];
        Iterator<String> iterator = iterator();
        for (int i = 0; i < objects.length; i++) {
            objects[i] = iterator.next();
        }
        return objects;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <U> U[] toArray(U[] a) {
        int size = size();
        U[] r = a.length >= size ? a :
                (U[]) newInstance(a.getClass().getComponentType(), size);
        Iterator<String> iterator = iterator();
        for (int i = 0; i < r.length; i++) {
            r[i] = (U) iterator.next();
        }
        return r;
    }

    public boolean add(String stringValue) {
        return wrapped.add(new StringValue(stringValue));
    }

    public boolean addEncrypted(String stringValue) {
        return wrapped.add(new EncryptedStringValue(stringValue));
    }

    @Override
    public boolean remove(Object o) {
        return wrapped.remove(stringToStringValue(o));
    }

    @Override
    public boolean containsAll(Collection<?> c) {
        return wrapped.containsAll(transformToStringValue(c));
    }

    @Override
    public boolean addAll(Collection<? extends String> c) {
        boolean changed = false;
        for (String s : c) {
            changed |= add(s);
        }
        return changed;
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        return wrapped.retainAll(transformToStringValue(c));
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        return wrapped.removeAll(transformToStringValue(c));
    }

    protected Collection<StringValue> transformToStringValue(Collection<?> c) {
        return Collections2.transform(c, new Function<Object, StringValue>() {
            @Override
            public StringValue apply(Object input) {
                if (input instanceof String) {
                    return new StringValue((String) input);
                } else if (input instanceof StringValue) {
                    return (StringValue) input;
                }
                throw new IllegalStateException("Cannot convert input [" + input + "]");
            }
        });
    }

    protected Object stringToStringValue(Object o) {
        if (o instanceof String) {
            return new StringValue((String) o);
        }
        return o;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj != null && obj instanceof AbstractStringView<?>) {
            @SuppressWarnings("unchecked")
            AbstractStringView<Collection<StringValue>> object = (AbstractStringView<Collection<StringValue>>) obj;
            return new EqualsBuilder().append(wrapped, object.wrapped).isEquals();
        }
        
        return false;
    }
    
    @Override
    public int hashCode() {
        return wrapped.hashCode();
    }

    @Override
    public void clear() {
        wrapped.clear();
    }

    public T getWrapped() {
        return wrapped;
    }
}
