package com.xebialabs.deployit.core.xml;

import java.util.Collection;
import java.util.List;

import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.core.AbstractStringView;
import com.xebialabs.deployit.core.MapStringStringView;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.engine.xml.AbstractConfigurationItemConverter;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.util.PasswordEncrypter;

import static com.xebialabs.deployit.repository.StringValueConverter.valueToString;

/**
 * Class that writes {@link ConfigurationItem} objects to XML using xstream.
 *
 * Note that this class implements the "writing" part of the ConfigurationItemReaderWriter in server-api-impl.
 */
public class ConfigurationItemWriter extends AbstractConfigurationItemConverter {

    protected PasswordEncrypter passwordEncrypter;

    public ConfigurationItemWriter() {
        this.passwordEncrypter = PasswordEncrypter.getInstance();
    }

    @Override
    protected void writeStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        if (propertyDescriptor.isPassword()) {
            writer.setValue(passwordEncrypter.ensureEncrypted(value.toString()));
        } else {
            super.writeStringProperty(value, propertyDescriptor, writer);
        }
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    protected void writeCollectionOfStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        if (value instanceof AbstractStringView) {
            value = Collections2.transform(((AbstractStringView) value).getWrapped(), valueToString(passwordEncrypter, true));
        }
        super.writeCollectionOfStringProperty(value, propertyDescriptor, writer);
    }

    @Override
    protected void writeMapStringStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        if (value instanceof MapStringStringView) {
            value = Maps.transformValues(((MapStringStringView) value).getWrapped(), valueToString(passwordEncrypter, true));
        }
        super.writeMapStringStringProperty(value, propertyDescriptor, writer);
    }

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        Object toBeMarshalled = source;
        if (source instanceof ValidatedConfigurationItem) {
            toBeMarshalled = ((ValidatedConfigurationItem) source).getWrapped();
        }
        super.marshal(toBeMarshalled, writer, context);

        if (source instanceof ValidatedConfigurationItem) {
            List<ValidationMessage> validations = ((ValidatedConfigurationItem) source).getValidations();
            writer.startNode("validation-messages");
            for (ValidationMessage validation : validations) {
                writer.startNode("validation-message");
                writer.addAttribute("ci", validation.getCiId());
                if (validation.getPropertyName() != null) {
                    writer.addAttribute("property", validation.getPropertyName());
                }
                writer.setValue(validation.getMessage());
                writer.endNode();
            }
            writer.endNode();
        }
    }

    @Override
    protected void writeCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        // To the remote side, we only write an id.
        writer.addAttribute("ref", ((ConfigurationItem) value).getId());
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void writeCollectionOfCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        // To the remote side, we only write ids
        for (ConfigurationItem configurationItem : (Collection<ConfigurationItem>) value) {
            writer.startNode("ci");
            writer.addAttribute("ref", configurationItem.getId());
            writer.endNode();
        }
    }

    /*
     * The following methods are both noops -- this class can not be used to read CIs from XML.
     */

    @Override
    protected void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
    }

    @Override
    protected void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
    }

}
