package com.xebialabs.deployit.jcr;

import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DEPLOYIT_NAMESPACE_PREFIX;
import static com.xebialabs.deployit.jcr.JcrConstants.DEPLOYIT_NAMESPACE_URI;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ROLES_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ROLE_ASSIGNMENTS_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.SECURITY_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.STEP_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASKS_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_NAME;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.ENVIRONMENTS;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.INFRASTRUCTURE;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;

import javax.jcr.NamespaceException;
import javax.jcr.NamespaceRegistry;
import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.nodetype.NodeType;
import javax.jcr.nodetype.NodeTypeManager;
import javax.jcr.nodetype.NodeTypeTemplate;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;

import org.apache.jackrabbit.api.JackrabbitRepository;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.security.principal.EveryonePrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;
import org.xml.sax.InputSource;

import com.xebialabs.deployit.ReleaseInfo;
import com.xebialabs.deployit.repository.internal.Root;


public class JackrabbitRepositoryFactoryBean implements InitializingBean, FactoryBean<Repository>, DisposableBean {

    private Resource homeDir;

    private Resource configuration;

    private JackrabbitRepository repository;

    private boolean autoInitialize;

    @Override
    public void afterPropertiesSet() throws IOException, RepositoryException {
        logger.info("Starting JCR repository");

        InputSource configurationInputSource = new InputSource(configuration.getInputStream());
        File homeDirFile = homeDir.getFile();
        if (!homeDirFile.exists() && !autoInitialize) {
            throw new RepositoryException("Jackrabbit home dir " + homeDirFile
                    + " does not exist. Please run the Deployit server with -setup -reinitialize to reinitialize the Deployit repository.");
        }

        RepositoryConfig repositoryConfig = RepositoryConfig.create(configurationInputSource, homeDirFile.getAbsolutePath());
        repository = RepositoryImpl.create(repositoryConfig);
        initializeJcrRepository();
        
        logger.debug("Started JCR repository");

    }

    private void initializeJcrRepository() {
        new JcrTemplate(repository).execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(final Session session) throws RepositoryException {
                final NamespaceRegistry namespaceRegistry = session.getWorkspace().getNamespaceRegistry();

                try {
                    // Attempt to get the URI for the "deployit" namespace and catch the NamespaceException that gets thrown if it has not been registered.
                    namespaceRegistry.getURI(DEPLOYIT_NAMESPACE_PREFIX);
                    // Return when no exception was thrown, i.e. nothing has to be done
                    return null;
                } catch (NamespaceException exc) {
                    if (!autoInitialize) {
                        throw new RepositoryException(
                                "The Deployit repository has not been initialized. Please run the Deployit server with -setup -reinitialize to reinitialize the Deployit repository.");
                    }
                }

                logger.info("Initializing JCR repository");

                namespaceRegistry.registerNamespace(DEPLOYIT_NAMESPACE_PREFIX, DEPLOYIT_NAMESPACE_URI);

                final NodeTypeManager typeManager = session.getWorkspace().getNodeTypeManager();
                createMixinNodeType(typeManager, CONFIGURATION_ITEM_NODETYPE_NAME);
                createMixinNodeType(typeManager, ARTIFACT_NODETYPE_NAME);

                createRoot(session, APPLICATIONS.getRootNodeName());
                createRoot(session, INFRASTRUCTURE.getRootNodeName());
                createRoot(session, ENVIRONMENTS.getRootNodeName());
                createRoot(session, CONFIGURATION.getRootNodeName());

                createMixinNodeType(typeManager, TASK_NODETYPE_NAME);
                createMixinNodeType(typeManager, STEP_NODETYPE_NAME);
                createNode(session, TASKS_NODE_NAME);

                createMixinNodeType(typeManager, CONFIGURATION_NODETYPE_NAME);
                createNode(session, CONFIGURATION_NODE_NAME, CONFIGURATION_NODETYPE_NAME);

                createNode(session, SECURITY_NODE_NAME);

                createNode(session, ROLES_NODE_NAME);
                createNode(session, ROLE_ASSIGNMENTS_NODE_NAME);

                setAccessControlOnRootNode(session);
                setRepoVersion(session);

                logger.debug("Initialized JCR repository");
                return null;
            }

        });
    }

    private void setRepoVersion(Session session) throws RepositoryException {
        Node node = session.getRootNode().addNode(VERSIONS_NODE_NAME);
        node.setProperty("deployit", ReleaseInfo.getReleaseInfo().getVersion());
        session.save();
    }

    private void createMixinNodeType(final NodeTypeManager typeManager, final String nodetypeName) throws RepositoryException {
        final NodeTypeTemplate nodeTypeTemplate = typeManager.createNodeTypeTemplate();
        nodeTypeTemplate.setName(nodetypeName);
        nodeTypeTemplate.setQueryable(true);
        nodeTypeTemplate.setAbstract(false);
        nodeTypeTemplate.setMixin(true);

        typeManager.registerNodeType(nodeTypeTemplate, false);
    }

    private void createRoot(final Session session, final String rootNodeName) throws RepositoryException {
        Node node = session.getRootNode().addNode(rootNodeName);
        node.addMixin(JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME);
        node.addMixin(NodeType.MIX_VERSIONABLE);
        node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "internal." + Root.class.getSimpleName());
        node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());

        session.save();
    }

    private void createNode(final Session session, final String nodeName, final String... mixinNodeTypeNames) throws RepositoryException {
        Node rootNode = session.getRootNode();
        Node addedNode = rootNode.addNode(nodeName);
        for (String each : mixinNodeTypeNames) {
            addedNode.addMixin(each);
        }

        session.save();
    }

    protected void setAccessControlOnRootNode(final Session session) throws RepositoryException {
        AccessControlManager acm = session.getAccessControlManager();
        AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies("/");

        boolean policySet = false;
        while (!policySet && applicablePolicies.hasNext()) {
            AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
            if (!(each instanceof JackrabbitAccessControlList))
                continue;

            JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;

            Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

            final EveryonePrincipal everyone = EveryonePrincipal.getInstance();
            for (AccessControlEntry entry : acl.getAccessControlEntries()) {
                if (entry.getPrincipal().equals(everyone)) {
                    acl.removeAccessControlEntry(entry);
                }
            }

            acl.addEntry(everyone, new Privilege[]{readPrivilege}, true);

            logger.debug("Setting {} on root", acl);
            acm.setPolicy("/", acl);
        }
    }

    @Override
    public Repository getObject() {
        return repository;
    }

    @Override
    public void destroy() {
        repository.shutdown();
    }

    @Override
    public Class<Repository> getObjectType() {
        return Repository.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    public Resource getHomeDir() {
        return homeDir;
    }

    @Required
    public void setHomeDir(final Resource homeDir) {
        this.homeDir = homeDir;
    }

    public Resource getConfiguration() {
        return configuration;
    }

    @Required
    public void setConfiguration(final Resource configuration) {
        this.configuration = configuration;
    }

    public boolean isAutoInitialize() {
        return autoInitialize;
    }

    public void setAutoInitialize(boolean autoInitialize) {
        this.autoInitialize = autoInitialize;
    }

    private static final Logger logger = LoggerFactory.getLogger(JackrabbitRepositoryFactoryBean.class);

}
