package com.xebialabs.deployit.jcr;

import javax.jcr.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.DeployitConfiguration;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException;

import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_USERNAME;


@Component
public class JcrTemplate {

    private final Repository repository;

    private static final ThreadLocal<Session> SESSION_STORE = new ThreadLocal<Session>();
    private static final Logger logger = LoggerFactory.getLogger(JcrTemplate.class);

    @Autowired
    public JcrTemplate(final Repository repository) {
        this.repository = repository;
    }

    public final <T> T execute(final JcrCallback<T> callback) throws RuntimeRepositoryException {
        Session session = SESSION_STORE.get();
        boolean createNewSession = false;
        if (session == null) {
            logger.debug("Did not find an existing session, creating a session for one JCR request.");
            createNewSession = true;
        } else if (!session.isLive()) {
            logger.debug("Existing session is no longer live, creating a session for one JCR request.");
            createNewSession = true;
        }
        if (createNewSession) {
            try {
                SimpleCredentials credentials = new SimpleCredentials(ADMIN_USERNAME, DeployitConfiguration.getInstance().getAdminPassword().toCharArray());
                session = repository.login(credentials);
                try {
                    return callback.doInJcr(session);
                } finally {
                    session.logout();
                }
            } catch (AccessDeniedException ade) {
                throw PermissionDeniedException.withMessage("Failed to complete your request.", ade);
            } catch (LoginException le) {
                throw new AuthenticationFailureException(le, "Incorrect credentials for user %s, did you forget to update the password in the deployit configuration file?", ADMIN_USERNAME);
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException(e.toString(), e);
            }
        } else {
            try {
                return callback.doInJcr(session);
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException(e.toString(), e);
            }
        }
    }

    public void login() {
        Credentials cred = new SimpleCredentials(ADMIN_USERNAME, DeployitConfiguration.getInstance().getAdminPassword().toCharArray());
        try {
            Session session = repository.login(cred);
            SESSION_STORE.set(session);
        } catch (AccessDeniedException ade) {
            throw PermissionDeniedException.withMessage("Failed to complete your request.", ade);
        } catch (LoginException le) {
            throw new AuthenticationFailureException(le, "Couldn't login to the repository", le);
        } catch (RepositoryException e) {
            throw new RuntimeRepositoryException(e.toString(), e);
        }
    }

    public boolean isLoggedIn() {
        return SESSION_STORE.get() != null && SESSION_STORE.get().isLive();
    }

    public void logout() {
        if (SESSION_STORE.get() != null) {
            SESSION_STORE.get().logout();
            SESSION_STORE.remove();
        } else {
            logger.warn("Superfluous logout call found.", new RuntimeException());
        }
    }
}
