package com.xebialabs.deployit.repository;

import java.util.*;

import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import javax.jcr.version.VersionManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.spi.event.CiMovedEvent;
import com.xebialabs.deployit.engine.spi.event.CiRenamedEvent;
import com.xebialabs.deployit.engine.spi.event.CisCreatedEvent;
import com.xebialabs.deployit.engine.spi.event.CisDeletedEvent;
import com.xebialabs.deployit.engine.spi.event.CisUpdatedEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.deployit.util.Tuple;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newTreeMap;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static com.xebialabs.deployit.repository.SearchQueryBuilder.CI_SELECTOR_NAME;
import static org.apache.jackrabbit.JcrConstants.JCR_ROOTVERSION;

@Component
public class JcrRepositoryService implements RepositoryService {

    private final JcrTemplate jcrTemplate;
    private final PasswordEncrypter passwordEncrypter;

    @Autowired
    public JcrRepositoryService(JcrTemplate jcrTemplate, PasswordEncrypter passwordEncrypter) {
        this.jcrTemplate = jcrTemplate;
        this.passwordEncrypter = passwordEncrypter;
    }

    @Override
    public boolean exists(final String id) {
        logger.debug("Checking whether node {} exists.", id);

        return jcrTemplate.execute(new JcrCallback<Boolean>() {
            @Override
            public Boolean doInJcr(final Session session) throws RepositoryException {
                return session.itemExists(getAbsolutePathFromId(id));
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id) {
        logger.debug("Reading node {}.", id);
        return this.<T>read(id, null, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, WorkDir workDir) {
        logger.debug("Reading node {} with workdir {}.", id, workDir);
        return this.<T>read(id, workDir, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final boolean useCache) {
        return this.<T>read(id, null, useCache);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final WorkDir workDir, final boolean useCache) {
        checkNotNull(id, "id is null");

        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(Session session) throws RepositoryException {
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    if (useCache) {
                        return NodeReader.<T>read(session, node, workDir, passwordEncrypter);
                    } else {
                        return NodeReader.<T>read(session, node, workDir, new NodeReaderContext(), passwordEncrypter);
                    }
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException(exc, "Repository entity [%s] not found", id);
                }
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> void create(final T... entities) {
        ChangeSet changeset = new ChangeSet();
        changeset.setCreateCis(Lists.<ConfigurationItem>newArrayList(entities));
        execute(changeset);

        EventBusHolder.publish(new CisCreatedEvent(entities));
    }

    @Override
    public <T extends ConfigurationItem> void move(final T entity, final String newId) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.addMoveCi(entity, newId);
        execute(changeSet);

        EventBusHolder.publish(new CiMovedEvent(entity, newId));
    }

    @Override
    public void rename(final String id, final String newName) {
        checkArgument(newName.indexOf('/') == -1, "New name [%s] should not contain a /", newName);
        ChangeSet changeSet = new ChangeSet();
        changeSet.addRenameCi(id, newName);
        execute(changeSet);

        EventBusHolder.publish(new CiRenamedEvent(id, newName));
    }

    @Override
    public <T extends ConfigurationItem> void createOrUpdate(T... entities) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.createOrUpdate(newArrayList(entities));
        execute(changeSet);
    }

    @Override
    public <T extends ConfigurationItem> void update(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        ArrayList<ConfigurationItem> updatedCis = Lists.<ConfigurationItem>newArrayList(cis);
        changeset.setUpdateCis(updatedCis);
        execute(changeset);

        EventBusHolder.publish(new CisUpdatedEvent(updatedCis));
    }

    @Override
    public void delete(final String... ids) {
        ChangeSet changeset = new ChangeSet();
        changeset.setDeleteCiIds(Lists.<String>newArrayList(ids));
        checkReferentialIntegrity(changeset);
        execute(changeset);

        EventBusHolder.publish(new CisDeletedEvent(ids));
    }

    @Override
    public void execute(ChangeSet changeset) {
        jcrTemplate.execute(new ChangeSetExecutor(changeset, passwordEncrypter));
    }

    @Override
    public void checkReferentialIntegrity(final ChangeSet changeset) throws ItemInUseException, ItemAlreadyExistsException {
        jcrTemplate.execute(new ReferentialIntegrityChecker(changeset, passwordEncrypter));
    }

    @Override
    public List<ConfigurationItemData> list(final SearchParameters parameters) {
        logger.debug("Listing node IDs with parameters {}.", parameters);

        checkNotNull(parameters);

        List<ConfigurationItem> entities = listEntities(parameters);
        return Lists.transform(entities, new Function<ConfigurationItem, ConfigurationItemData>() {
            public ConfigurationItemData apply(ConfigurationItem input) {
                return new ConfigurationItemData(input.getId(), input.getType());
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> List<T> listEntities(final SearchParameters parameters) {
        logger.debug("Listing nodes with parameters {}.", parameters);

        checkNotNull(parameters, "parameters is null");
        return jcrTemplate.execute(new JcrCallback<List<T>>() {
            @Override
            public List<T> doInJcr(final Session session) throws RepositoryException {
                final Query query = new SearchQueryBuilder(parameters).build(session);
                final QueryResult queryResult = query.execute();
                return getOrderedEntities(parameters, session, queryResult);
            }
        });
    }

    private <T extends ConfigurationItem> List<T> getOrderedEntities(final SearchParameters parameters, final Session session, final QueryResult queryResult) throws RepositoryException {
        final List<T> items = newArrayList();
        final RowIterator iterator = queryResult.getRows();
        while (iterator.hasNext()) {
            Node node = iterator.nextRow().getNode(CI_SELECTOR_NAME);
            // Your IDE might not think <T> is needed here, but your compiler does ;)
            T item = NodeReader.<T>read(session, node, null, passwordEncrypter);
            items.add(item);
        }
        return items;
    }

    @Override
    public <T extends ConfigurationItem> List<T> getVersionHistory(final String id) {
        logger.debug("Retrieving version history for node {}.", id);

        return jcrTemplate.execute(new JcrCallback<List<T>>() {
            @Override
            public List<T> doInJcr(final Session session) throws RepositoryException {
                final List<T> items = newArrayList();
                final VersionHistory history = session.getWorkspace().getVersionManager().getVersionHistory(getAbsolutePathFromId(id));
                final VersionIterator versions = history.getAllLinearVersions();
                while (versions.hasNext()) {
                    Version version = versions.nextVersion();
                    if (version.getName().equals(JCR_ROOTVERSION)) {
                        continue;
                    }

                    Node node = version.getFrozenNode();
                    // Your IDE might not think <T> is needed here, but your compiler does ;)
                    T item = NodeReader.<T>read(session, node, null, passwordEncrypter);
                    // The ID set by the NodeReader will be the path of the frozen node minus the leading slash. Overwrite that with the ID the user expects.
                    item.setId(id);
                    items.add(item);
                }
                return items;
            }
        });
    }

    private static class ChangeSetExecutor implements JcrCallback<Object> {

        private final ChangeSet changeset;
        private final PasswordEncrypter passwordEncrypter;

        ChangeSetExecutor(ChangeSet changeset, PasswordEncrypter passwordEncrypter) {
            checkNotNull(changeset);
            this.changeset = changeset;
            this.passwordEncrypter = passwordEncrypter;
        }

        @Override
        public Object doInJcr(Session session) throws RepositoryException {
            execute(session, true);
            return null;
        }

        public void execute(Session session, boolean autocommit) throws RepositoryException {
            try {
                createEntities(changeset.getCreateCis(), session);
                createOrUpdateEntities(changeset.getCreateOrUpdateCis(), session);
                updateEntities(changeset.getUpdateCis(), session);
                moveEntities(changeset.getMoveCis(), session);
                renameEntities(changeset.getRenameCis(), session);
                SortedMap<String, VersionHistory> versionHistories = deleteEntities(changeset.getDeleteCiIds(), session);
                if (autocommit) {
                    saveSession(session);
                    removeVersionHistories(versionHistories);
                }
            } catch (ReferentialIntegrityException exc) {
                throw new ItemInUseException(exc, "Cannot delete configuration items [%s] because one of the configuration items, or one of their children, is still being referenced", on(',').join(changeset.getDeleteCiIds()));
            }
        }

        private void createOrUpdateEntities(List<ConfigurationItem> createOrUpdateCis, Session session) throws RepositoryException {
            List<ConfigurationItem> toBeCreated = newArrayList();
            List<ConfigurationItem> toBeUpdated = newArrayList();

            for (ConfigurationItem createOrUpdateCi : createOrUpdateCis) {
                if (!session.nodeExists(getAbsolutePathFromId(createOrUpdateCi.getId()))) {
                    toBeCreated.add(createOrUpdateCi);
                } else {
                    toBeUpdated.add(createOrUpdateCi);
                }
            }

            createEntities(toBeCreated, session);
            updateEntities(toBeUpdated, session);
        }

        private void createEntities(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            if (!entities.isEmpty()) {
                logger.trace("Creating nodes...");
            }

            verifyEntitiesMeetCreationPreconditions(entities);
            Collections.sort(entities, new SortHierarchyComparator());
            List<NodeWriter> writers = newArrayList();
            for (ConfigurationItem entity : entities) {
                Node node = createNode(entity, session);
                final NodeWriter nodeWriter = new NodeWriter(session, entity, node, passwordEncrypter);
                nodeWriter.writeBasics();
                writers.add(nodeWriter);
            }
            writeEntities(writers);
        }

        private Node createNode(ConfigurationItem entity, Session session) throws RepositoryException {
            validateNodeStoredInCorrectPath(entity.getId(), entity.getType(), session);
            logger.debug("Creating node {}.", entity.getId());
            final Node node = session.getRootNode().addNode(entity.getId());
            int index = node.getIndex();
            if (index != 1) {
                node.remove();
                throw new ItemAlreadyExistsException("A Configuration Item with ID [%s] already exists.", entity.getId());
            }
            setMixins(entity, node);
            return node;
        }

        private void updateEntities(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            if (!entities.isEmpty()) {
                logger.trace("Updating nodes...");
            }

            List<Node> nodes = retrieveAndCheckpointNodesToBeUpdated(entities, session);
            for (int i = 0; i < entities.size(); i++) {
                updateNode(entities.get(i), nodes.get(i), session);
            }
        }

        private List<Node> retrieveAndCheckpointNodesToBeUpdated(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (ConfigurationItem entity : entities) {
                checkNotNull(entity, "entity is null");
                checkNotNull(entity.getId(), "id is null");
                try {
                    final Node node = session.getNode(getAbsolutePathFromId(entity.getId()));
                    checkpoint(node, session);
                    nodes.add(node);
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException("Repository entity [%s] not found", entity.getId());
                }
            }
            return nodes;
        }

        private void updateNode(ConfigurationItem entity, Node node, Session session) throws RepositoryException {
            logger.debug("Updating node {}.", entity.getId());
            new NodeWriter(session, entity, node, passwordEncrypter).write();
        }

        private void moveEntities(List<Tuple<ConfigurationItem, String>> moveCis, Session session) throws RepositoryException {
            if (!moveCis.isEmpty()) {
                logger.debug("Moving nodes...");
            }

            for (Tuple<ConfigurationItem, String> moveCi : moveCis) {
                ConfigurationItem toBeMoved = moveCi.getA();
                String newId = moveCi.getB();
                checkArgument(!toBeMoved.getId().equals(newId), "Cannot move ci [%s] to same location", newId);
                if (!toBeMoved.getType().equals(Type.valueOf(Directory.class))) {
                    checkArgument(!DescriptorRegistry.getDescriptor(toBeMoved.getType()).getRoot().equals(Metadata.ConfigurationItemRoot.NESTED), "Can only move 'rooted' cis [%s]", toBeMoved.getType());
                }

                validateNodeStoredInCorrectPath(newId, toBeMoved.getType(), session);
                moveNode(getAbsolutePathFromId(toBeMoved.getId()), getAbsolutePathFromId(newId), session);
            }
        }

        private void renameEntities(List<Tuple<String, String>> renameCis, Session session) throws RepositoryException {
            if (!renameCis.isEmpty()) {
                logger.trace("Renaming nodes...");
            }

            for (Tuple<String, String> renameCi : renameCis) {
                String oldId = renameCi.getA();
                String newId = oldId.substring(0, oldId.lastIndexOf('/') + 1) + renameCi.getB();
                moveNode(getAbsolutePathFromId(oldId), getAbsolutePathFromId(newId), session);
            }
        }

        private void moveNode(String fromId, String toId, Session session) throws RepositoryException {
            logger.debug("Moving/renaming node {} to {}", fromId, toId);
            if (session.nodeExists(toId)) {
                throw new ItemAlreadyExistsException("The destination id [%s] exists.", toId);
            }
            session.move(fromId, toId);
        }

        private SortedMap<String, VersionHistory> deleteEntities(List<String> entityIds, Session session) throws RepositoryException {
            if (entityIds.isEmpty()) {
                logger.trace("Deleting nodes...");
            }

            List<Node> nodes = retrieveNodesToBeDeleted(entityIds, session);
            SortedMap<String, VersionHistory> versionHistories = newTreeMap();
            for (Node node : nodes) {
                checkpointAndRetrieveVersionHistories(node, session, versionHistories);
            }

            for (Node node : nodes) {
                logger.debug("Deleting node {}.", getIdFromAbsolutePath(node.getPath()));
                node.remove();
            }

            return versionHistories;
        }

        private List<Node> retrieveNodesToBeDeleted(List<String> ids, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (String id : ids) {
                checkNotNull(id, "id is null");
                logger.trace("Retrieving node {} to delete it.", id);
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    nodes.add(node);
                } catch (PathNotFoundException ignored) {
                    //ignore
                }
            }
            return nodes;
        }

        private void checkpointAndRetrieveVersionHistories(Node node, Session session, SortedMap<String, VersionHistory> versionHistories) throws RepositoryException {
            // Create a checkpoint so that there is at least one version. Jackrabbit will only purge empty version histories that had at least one
            // non-baseVersion.
            checkpoint(node, session);

            final String id = getIdFromAbsolutePath(node.getPath());
            logger.trace("Retrieving version history for node {} to delete it.", id);
            final VersionHistory versionHistory = session.getWorkspace().getVersionManager().getVersionHistory(node.getPath());
            versionHistories.put(id, versionHistory);
            NodeIterator nodes = node.getNodes();
            while (nodes.hasNext()) {
                Node each = nodes.nextNode();
                checkpointAndRetrieveVersionHistories(each, session, versionHistories);
            }
        }

        private void checkpoint(Node node, Session session) throws RepositoryException {
            logger.trace("Checkpointing node {}.", getIdFromAbsolutePath(node.getPath()));
            VersionManager versionManager = session.getWorkspace().getVersionManager();
            versionManager.checkpoint(node.getPath());
        }

        private void saveSession(Session session) throws RepositoryException {
            logger.trace("Committing JCR session");
            session.save();
        }

        private void removeVersionHistories(SortedMap<String, VersionHistory> versionHistories) throws RepositoryException {
            logger.trace("Removing version histories that have become obsolete because their corresponding nodes or one of their ancestors was deleted.");
            for (Map.Entry<String, VersionHistory> entry : versionHistories.entrySet()) {
                removeVersionHistory(entry.getKey(), entry.getValue());
            }
        }

        private void removeVersionHistory(String id, VersionHistory versionHistory) throws RepositoryException {
            logger.debug("Removing version history for node {}.", id);
            final VersionIterator allVersions = versionHistory.getAllVersions();
            while (allVersions.hasNext()) {
                Version version = allVersions.nextVersion();
                if (!version.getName().equals(JCR_ROOTVERSION)) {
                    logger.trace("Removing version {} from version history for node {}.", version.getName(), id);
                    versionHistory.removeVersion(version.getName());
                }
            }
        }

        private void verifyEntitiesMeetCreationPreconditions(List<ConfigurationItem> cis) {
            for (int i = 0; i < cis.size(); i++) {
                ConfigurationItem ci = cis.get(i);
                checkNotNull(ci, "ci at index %s is null.", i);
                checkNotNull(ci.getId(), "ci at index %s has null id.", i);

                if (ci instanceof SourceArtifact) {
                    checkArgument(((SourceArtifact) ci).getFile() != null, "Artifact %s should have file", ci.getId());
                }
            }
        }

        private void validateNodeStoredInCorrectPath(String id, Type type, Session session) throws RepositoryException {
            checkArgument(DescriptorRegistry.exists(type), "Unknown configuration item type %s", type);

            final Descriptor desc = DescriptorRegistry.getDescriptor(type);

            final String[] pathElements = id.split("/");

            if (desc.isAssignableTo(Directory.class)) {
                checkArgument(pathElements.length >= 2, "A core.Directory must be stored under a Root node, not under %s.", id);
                for (Metadata.ConfigurationItemRoot configurationItemRoot : Metadata.ConfigurationItemRoot.values()) {
                    if (configurationItemRoot.getRootNodeName().equals(pathElements[0])) {
                        return;
                    }
                }
                throw new Checks.IncorrectArgumentException("A core.Directory must be stured under a valid Root node, not under %s.", id);
            } else if (desc.getRoot() != NESTED) {
                checkArgument(pathElements[0].equals(desc.getRoot().getRootNodeName()), "Configuration item of type %s cannot be stored at %s. It should be stored under %s", type, id, desc.getRoot().getRootNodeName());
            } else {
                checkArgument(pathElements.length >= 3, "Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node", type, id);
                final String parentId = id.substring(0, id.lastIndexOf('/'));
                final Node parentNode = session.getNode(getAbsolutePathFromId(parentId));
                checkArgument(parentNode != null, "Configuration item of type %s cannot be stored at %s. The parent does not exist", type, id);
                checkArgument(parentNode.isNodeType(CONFIGURATION_ITEM_NODETYPE_NAME), "Configuration item of type %s cannot be stored at %s. The parent is not a configuration item", type, id);
                final String parentTypeName = parentNode.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
                final Descriptor parentDescriptor = DescriptorRegistry.getDescriptor(parentTypeName);

                // Check parent relation
                for (PropertyDescriptor aPD : desc.getPropertyDescriptors()) {
                    if (aPD.getKind() == CI && aPD.isAsContainment()) {
                        if (parentDescriptor.isAssignableTo(aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                // Check child relation
                for (PropertyDescriptor aPD : parentDescriptor.getPropertyDescriptors()) {
                    if (aPD.getKind() == SET_OF_CI && aPD.isAsContainment()) {
                        if (desc.isAssignableTo(aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                throw new Checks.IncorrectArgumentException("Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type", type, id);
            }
        }

        private void setMixins(ConfigurationItem entity, Node node)
                throws RepositoryException {
            if (entity instanceof Artifact) {
                node.addMixin(ARTIFACT_NODETYPE_NAME);
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            } else {
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            }
            node.addMixin(NodeType.MIX_REFERENCEABLE);
            node.addMixin(NodeType.MIX_VERSIONABLE);
        }

        private void writeEntities(List<NodeWriter> writers) throws RepositoryException {
            for (NodeWriter writer : writers) {
                writer.write();
            }
        }

    }

    private static class SortHierarchyComparator implements Comparator<ConfigurationItem> {
        @Override
        public int compare(ConfigurationItem o1, ConfigurationItem o2) {
            final int nrSlashes1 = StringUtils.countOccurrencesOf(o1.getId(), "/");
            final int nrSlashes2 = StringUtils.countOccurrencesOf(o2.getId(), "/");
            return nrSlashes1 - nrSlashes2;
        }
    }

    private static class ReferentialIntegrityChecker implements JcrCallback<Object> {

        private final ChangeSet changeset;
        private final PasswordEncrypter passwordEncrypter;

        ReferentialIntegrityChecker(ChangeSet changeset, PasswordEncrypter passwordEncrypter) {
            checkNotNull(changeset);
            this.changeset = changeset;
            this.passwordEncrypter = passwordEncrypter;
        }

        @Override
        public Object doInJcr(Session session) throws RepositoryException {
            checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(session);
            Multimap<String, PropertyRef> deletedEntityRefs = findReferencesToDeletedNodes(session);
            ChangeSetExecutor changeSetExecutor = new ChangeSetExecutor(changeset, passwordEncrypter);
            changeSetExecutor.execute(session, false);
            try {
                checkForRemainingReferencesToDeletedNodes(deletedEntityRefs, session);
            } finally {
                session.refresh(false);
            }
            return null;
        }

        private void checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(Session session) throws RepositoryException {
            for (ConfigurationItem entity : changeset.getCreateCis()) {
                if (session.nodeExists(getAbsolutePathFromId(entity.getId()))) {
                    throw new ItemAlreadyExistsException("Repository entity %s already exists", entity.getId());
                }
            }
        }

        private Multimap<String, PropertyRef> findReferencesToDeletedNodes(Session session) throws RepositoryException {
            Multimap<String, PropertyRef> deletedEntityRefs = ArrayListMultimap.create();
            for (String deleteNodeId : changeset.getDeleteCiIds()) {
                if (!session.nodeExists(getAbsolutePathFromId(deleteNodeId))) {
                    continue;
                }
                Node node = session.getNode(getAbsolutePathFromId(deleteNodeId));
                PropertyIterator references = node.getReferences();
                while (references.hasNext()) {
                    Property property = references.nextProperty();
                    Node parent = property.getParent();
                    deletedEntityRefs.put(deleteNodeId, new PropertyRef(parent.getPath(), property.getName(), node.getIdentifier()));
                }
            }
            return deletedEntityRefs;
        }

        private void checkForRemainingReferencesToDeletedNodes(Multimap<String, PropertyRef> deletedEntityRefs, Session session) throws RepositoryException {
            for (String deleteNodeId : deletedEntityRefs.keySet()) {
                for (PropertyRef propertyRef : deletedEntityRefs.get(deleteNodeId)) {
                    if (session.nodeExists(propertyRef.owningNode)) {
                        Node owningNode = session.getNode(propertyRef.owningNode);
                        if (owningNode.hasProperty(propertyRef.property)) {
                            Property property = owningNode.getProperty(propertyRef.property);
                            Value[] values = property.isMultiple() ? property.getValues() : new Value[]{property.getValue()};
                            for (Value value : values) {
                                if (value.getString().equals(propertyRef.uuidOfReferencedNode)) {
                                    throw new ItemInUseException("Repository entity %s is still referenced by %s", deleteNodeId, propertyRef.owningNode);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private static class PropertyRef {
        private String owningNode;
        private String property;
        private String uuidOfReferencedNode;

        PropertyRef(String owningNode, String property, String uuidOfReferencedNode) {
            this.owningNode = owningNode;
            this.property = property;
            this.uuidOfReferencedNode = uuidOfReferencedNode;
        }
    }

    private static Logger logger = LoggerFactory.getLogger(JcrRepositoryService.class);

}
