package com.xebialabs.deployit.repository;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.jcr.*;

import org.apache.jackrabbit.value.ReferenceValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.core.AbstractStringView;
import com.xebialabs.deployit.core.MapStringStringView;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.io.Imploder;
import com.xebialabs.deployit.jcr.JcrUtils;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TOKEN_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

/**
 * Writes an {@link ConfigurationItem} to a JCR {@link Node}.
 */
class NodeWriter {

    private final Session session;
    private final ConfigurationItem item;
    private final Node node;
    private boolean basicsWritten = false;
    private final PasswordEncrypter passwordEncrypter;

    NodeWriter(final Session session, final ConfigurationItem item, final Node node, PasswordEncrypter passwordEncrypter) {
        this.session = session;
        this.item = item;
        this.node = node;
        this.passwordEncrypter = passwordEncrypter;
    }

    void writeBasics() throws RepositoryException {
        node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, item.getType().toString());
        if (item instanceof BaseConfigurationItem) {
            String token = ((BaseConfigurationItem) item).get$token();
            if (Strings.isNullOrEmpty(token)
                    || (node.hasProperty(TOKEN_PROPERTY_NAME) && node.getProperty(TOKEN_PROPERTY_NAME).getString().equals(token))) {
                node.setProperty(TOKEN_PROPERTY_NAME, UUID.randomUUID().toString());
            } else {
                throw new ItemConflictException("Repository entity [%s] has been updated since you read it. Please reload the CI from the repository again.", item.getId());
            }
        }
        basicsWritten = true;
    }

    void write() throws RepositoryException {
        if (!basicsWritten) {
            writeBasics();
        }
        copyData();
        copyValuesIntoNode();
        copyMetadata();
    }

    private void copyData() throws RepositoryException {
        if (item instanceof SourceArtifact) {
            OverthereFile file = ((Artifact) item).getFile();
            if (file != null) {
                node.setProperty(FILENAME_PROPERTY_NAME, file.getName());
                InputStream dataInput;
                if (item instanceof FolderArtifact && file.isDirectory()) {
                    try {
                        byte[] implode = Imploder.implode(file);
                        dataInput = new ByteArrayInputStream(implode);
                    } catch (IOException e) {
                        throw new RuntimeIOException("Could not zip up the data in file " + file);
                    }
                } else {
                    dataInput = file.getInputStream();
                }
                try {
                    Binary binary = session.getValueFactory().createBinary(dataInput);
                    node.setProperty(DATA_PROPERTY_NAME, binary);
                } finally {
                    closeQuietly(dataInput);
                }
            }
        }
    }

    private void copyValuesIntoNode() throws RepositoryException {
        Descriptor ciDescriptor = DescriptorRegistry.getDescriptor(item.getType());
        Collection<PropertyDescriptor> propertyDescriptors = Collections2.filter(ciDescriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
            @Override
            public boolean apply(PropertyDescriptor input) {
                return !input.isTransient();
            }
        });
        for (PropertyDescriptor pd : propertyDescriptors) {
            if (pd.get(item) == null) {
                removePropertyFromNode(pd);
                continue;
            }

            logger.trace("Writing property [{}] with value [{}]", pd.getFqn(), pd.isPassword() ? "********" : pd.get(item));

            switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
            case ENUM:
                copyPrimitivePropertyIntoNode(pd);
                break;
            case SET_OF_STRING:
                copyCollectionOfStringsPropertyIntoNode(pd);
                break;
            case SET_OF_CI:
                copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
                break;
            case LIST_OF_STRING:
                copyCollectionOfStringsPropertyIntoNode(pd);
                break;
            case LIST_OF_CI:
                copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
                break;
            case CI:
                copyConfigurationItemPropertyIntoNode(pd);
                break;
            case MAP_STRING_STRING:
                copyMapPropertyIntoNode(pd);
                break;
            default:
                throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported kind " + pd.getKind());

            }
        }
    }

    @SuppressWarnings("unchecked")
    private void copyMapPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        Map<String, String> value = (Map<String, String>) pd.get(item);
        Map<String, String> toWrite = value;
        if (value instanceof MapStringStringView) {
            toWrite = Maps.transformValues(((MapStringStringView) value).getWrapped(), StringValueConverter.valueToString(passwordEncrypter));
        }
        JcrUtils.writeMap(node, pd.getName(), toWrite);
    }

    private void copyMetadata() throws RepositoryException {
        node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());
    }

    private void removePropertyFromNode(PropertyDescriptor pd) throws RepositoryException {
        try {
            node.getProperty(pd.getName()).remove();
        } catch (PathNotFoundException ignored) {
        }
    }

    private void copyPrimitivePropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        String valueAsString = pd.get(item).toString();
        if(pd.getKind() == STRING && pd.isPassword()) {
            valueAsString = passwordEncrypter.ensureEncrypted(valueAsString);
        }
        node.setProperty(pd.getName(), valueAsString);
    }

    private void copyCollectionOfStringsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        final ValueFactory valueFactory = session.getValueFactory();
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);

        Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
        List<Value> jcrValueList = newArrayList();
        if (valueAsCollection instanceof AbstractStringView) {
            @SuppressWarnings("rawtypes")
            AbstractStringView view = (AbstractStringView) valueAsCollection;
            for (Object o : view.getWrapped()) {
                if (!(o instanceof StringValue)) {
                    throw new IllegalArgumentException("Element in property " + pd.getName() + " of configuration item " + item.getId()
                            + " is not a StringValue: " + o);
                }
                StringValue stringValue = (StringValue) o;
                jcrValueList.add(valueFactory.createValue(converter.convert(stringValue)));
            }
        } else {
            for (Object each : valueAsCollection) {
                if (!(each instanceof String)) {
                    throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + item.getId()
                            + " is not a String: " + each);
                }
                String stringValue = (String) each;
                jcrValueList.add(valueFactory.createValue(stringValue));
            }
        }
        Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);
        node.setProperty(pd.getName(), jcrValues);
    }

    private void copyConfigurationItemPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        if (pd.isAsContainment()) {
            return;
        }

        String referencedCiId = ((ConfigurationItem) pd.get(item)).getId();
        Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
        node.setProperty(pd.getName(), referencedCi);
    }

    private void copyCollectionOfConfigurationItemsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        if (pd.isAsContainment()) {
            return;
        }

        Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
        List<Value> jcrReferenceList = newArrayList();
        for (Object each : valueAsCollection) {
            if (!(each instanceof ConfigurationItem)) {
                throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + item.getId()
                        + " is not a ConfigurationItem: " + each);
            }
            String referencedCiId = ((ConfigurationItem) each).getId();
            Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
            jcrReferenceList.add(new ReferenceValue(referencedCi));
        }
        Value[] jcrReferenceValues = jcrReferenceList.toArray(new Value[jcrReferenceList.size()]);
        node.setProperty(pd.getName(), jcrReferenceValues);
    }

    private Collection<?> convertObjectValueToCollection(PropertyDescriptor pd, Object value) {
        if (!(value instanceof Collection)) {
            throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + item.getId() + " is not a Collection: " + value);
        }
        return (Collection<?>) value;
    }

    private static final Logger logger = LoggerFactory.getLogger(NodeWriter.class);
}
