package com.xebialabs.deployit.repository;

import com.google.common.base.Function;

import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.util.PasswordEncrypter;

public class StringValueConverter {

    public static final String ENCRYPTED_PREFIX = "e{";
    public static final String PLAIN_PREFIX = "u{";
    public static final String SUFFIX = "}";

    private final PasswordEncrypter passwordEncrypter;

    public StringValueConverter(PasswordEncrypter passwordEncrypter) {
        this.passwordEncrypter = passwordEncrypter;
    }

    public static final Function<StringValue, String> valueToString(PasswordEncrypter passwordEncrypter) {
        return valueToString(passwordEncrypter, false);
    }

    public static final Function<StringValue, String> valueToString(PasswordEncrypter passwordEncrypter, final boolean fallbackEncrypted) {
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);
        return new Function<StringValue, String>() {
            @Override
            public String apply(final StringValue input) {
                return converter.convert(input, fallbackEncrypted);
            }
        };
    }

    public static final Function<String, StringValue> stringToValue(PasswordEncrypter passwordEncrypter) {
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);
        return new Function<String, StringValue>() {
            @Override
            public StringValue apply(final String input) {
                return converter.convert(input);
            }
        };
    }

    public String convert(StringValue v, boolean keepPlainAsIs) {
        if (v instanceof EncryptedStringValue) {
            return ENCRYPTED_PREFIX + passwordEncrypter.ensureEncrypted(v.toString()) + SUFFIX;
        } else if (!keepPlainAsIs) {
            return PLAIN_PREFIX + v.toString() + SUFFIX;
        } else {
            return v.toString();
        }
    }

    String convert(StringValue v) {
        return convert(v, false);
    }

    public StringValue convert(String s) {
        return convert(s, false);
    }

    public StringValue convert(String s, boolean fallbackEncrypted) {
        if (s.startsWith(ENCRYPTED_PREFIX)) {
            String substring = s.substring(ENCRYPTED_PREFIX.length(), s.length() - SUFFIX.length());
            return new EncryptedStringValue(passwordEncrypter.ensureDecrypted(substring));
        } else if (s.startsWith(PLAIN_PREFIX)) {
            String substring = s.substring(PLAIN_PREFIX.length(), s.length() - SUFFIX.length());
            return new StringValue(substring);
        } else {
            if (fallbackEncrypted) {
                return new EncryptedStringValue(s);
            } else {
                // FALLBACK SCENARIO
                return new StringValue(s);
            }
        }
    }
}
