package com.xebialabs.deployit.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.JcrPathHelper;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.permission.Permission;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.jcr.JcrUtils.readMap;
import static com.xebialabs.deployit.jcr.JcrUtils.writeMap;

public class Permissions {
    static final Splitter PERMISSION_SPLITTER = Splitter.on(',').omitEmptyStrings();
    static final Joiner PERMISSION_JOINER = Joiner.on(",").skipNulls();
    
    static Iterable<String> splitPrincipals(String principals) {
        if (principals == null) return newArrayList();
        return PERMISSION_SPLITTER.split(principals);
    }
    
    static Iterable<Integer> splitRoles(String roles) {
        return Iterables.transform(splitPrincipals(roles), new Function<String, Integer>() {
            public Integer apply(String input) {
                return Integer.valueOf(input);
            }
        });
    }
    
    static String joinRoles(Iterable<Integer> roles) {
        return PERMISSION_JOINER.join(roles);
    }
    
    static String joinRoles(Integer... roles) {
        return PERMISSION_JOINER.join(roles);
    }
    
    static Iterable<Integer> rolesToIds(Iterable<Role> roles) {
        return Iterables.transform(roles, new Function<Role, Integer>() {
            public Integer apply(Role input) {
                return input.getId();
            }
        });
    }

    public static ImmutableMap<Integer, Role> buildLookup(List<Role> roles) {
        return Maps.uniqueIndex(roles, new Function<Role, Integer>() {
            @Override
            public Integer apply(Role input) {
                return input.getId();
            }
        });
    }

    static List<String> getFullTreeAsSeparateNodesInChildToRootOrder(String onConfigurationItem) {
        ArrayList<String> list = newArrayList();
        for (int index = onConfigurationItem.indexOf('/'); index > -1; index = onConfigurationItem.indexOf('/', index + 1)) {
            list.add(onConfigurationItem.substring(0, index));
        }
        list.add(onConfigurationItem);

        Collections.reverse(list);

        logger.debug("Split off tree: {} to {}", onConfigurationItem, list);

        return list;
    }

    public static Authentication getAuthentication() {
        return SecurityContextHolder.getContext().getAuthentication();
    }

    public static String getAuthenticatedUserName() {
        return getAuthentication() != null ? getAuthentication().getName() : null;
    }

    public static void clearSecurityContext() {
        SecurityContextHolder.clearContext();
    }

    public static Collection<String> authenticationToPrincipals(Authentication authentication) {
        final String name = authentication.getName();
        final Iterable<String> authorities = getAuthorities(authentication);
        final Collection<String> allPrincipals = newHashSet(authorities);
        allPrincipals.add(name);
        logger.trace("All principals for user [{}] are: {}", authentication.getName(), allPrincipals);
        return allPrincipals;
    }

    private static Iterable<String> getAuthorities(Authentication authentication) {
        return Iterables.transform(authentication.getAuthorities(), new Function<GrantedAuthority, String>() {
            @Override
            public String apply(GrantedAuthority input) {
                return input.getAuthority();
            }
        });
    }

    static Node readSecurable(String securable, Session session) throws RepositoryException {
        Node node = session.getNode(JcrPathHelper.getAbsolutePathFromId(securable));
        checkSecurable(node);
        return node;
    }

    static void checkSecurable(Node node) throws RepositoryException {
        String type = node.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
        checkArgument(isSecurable(node), "Cannot set permission on [%s] because it is not a Directory or Root [%s]", JcrPathHelper.getIdFromAbsolutePath(node.getPath()), type);
    }
    
    static boolean isSecurable(Node node) throws RepositoryException {
        String type = node.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
        return Type.valueOf(type).isSubTypeOf(Type.valueOf(Securable.class));
    }
    
    static Map<String, String> readPermissionMap(Node s) throws RepositoryException {
        Map<String, String> permissions = newHashMap();
        if (s.hasProperty(Securable.SECURITY_PERMISSIONS_PROPERTY)) {
            Property securityPermissions = s.getProperty(Securable.SECURITY_PERMISSIONS_PROPERTY);
            permissions = readMap(securityPermissions);
        }
        return permissions;
    }
    
    static void writePermissionMap(Node s, Map<String, String> permissions) throws RepositoryException {
        writeMap(s, Securable.SECURITY_PERMISSIONS_PROPERTY, permissions);
    }



    private static final Logger logger = LoggerFactory.getLogger(Permissions.class);

    public static Collection<Permission> isApplicableTo(Collection<Permission> values, final String id) {
        return Collections2.filter(values, new Predicate<Permission>() {
            public boolean apply(Permission input) {
                return !input.isApplicableTo(id);
            }
        });
    }
}
