package com.xebialabs.deployit.security.permission;

import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.security.permission.PermissionHandler.Level;

import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.BOTH;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.CI;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.GLOBAL;

public enum Permission {
    LOGIN("login", GLOBAL),
    READ("read", CI, new ReadPermissionHandler()),
    IMPORT_INITIAL("import#initial", CI, Metadata.ConfigurationItemRoot.APPLICATIONS),
    IMPORT_UPGRADE("import#upgrade", CI, Metadata.ConfigurationItemRoot.APPLICATIONS),
    IMPORT_REMOVE("import#remove", CI, Metadata.ConfigurationItemRoot.APPLICATIONS),
    DEPLOY_INITIAL("deploy#initial", CI, Metadata.ConfigurationItemRoot.ENVIRONMENTS),
    DEPLOY_UPGRADE("deploy#upgrade", CI, Metadata.ConfigurationItemRoot.ENVIRONMENTS),
    DISCOVERY("discovery", GLOBAL),
    UNDEPLOY("deploy#undeploy", CI, Metadata.ConfigurationItemRoot.ENVIRONMENTS),
    EDIT_REPO("repo#edit", CI),
    EDIT_SECURITY("security#edit", GLOBAL),
    TASK_MOVESTEP("task#move_step", BOTH, Metadata.ConfigurationItemRoot.ENVIRONMENTS),
    TASK_SKIPSTEP("task#skip_step", BOTH, Metadata.ConfigurationItemRoot.ENVIRONMENTS),
    TASK_ASSIGN("task#assign", GLOBAL),
    CONTROLTASK_EXECUTE("controltask#execute", GLOBAL),
    ADMIN("admin", GLOBAL);

    private PermissionHandler permissionHandler;
    private String permissionName;
    private Level level;
    private Metadata.ConfigurationItemRoot root;

    Permission(String permissionName, Level level) {
        this.permissionName = permissionName;
        this.level = level;
        this.permissionHandler = new PermissionHandler(Permission.this);
    }

    Permission(String permissionName, Level level, PermissionHandler permissionHandler) {
        this.permissionName = permissionName;
        this.level = level;
        this.permissionHandler = permissionHandler;
    }

    Permission(String permissionName, Level level, Metadata.ConfigurationItemRoot root) {
        this(permissionName, level);
        this.root = root;
    }

    public static Permission find(String permissionName) {
        for (Permission permission : values()) {
            if (permission.permissionName.equals(permissionName)) {
                return permission;
            }
        }
        return null;
    }

    public PermissionHandler getPermissionHandler() {
        return permissionHandler;
    }

    public String getPermissionName() {
        return permissionName;
    }

    public Level getLevel() {
        return level;
    }

    public Metadata.ConfigurationItemRoot getRoot() {
        return root;
    }

    public boolean isApplicableTo(String id) {
        id = nullToEmpty(id);
        boolean idIsGlobal = id.isEmpty() || id.equals(JcrConstants.GLOBAL_SECURITY_ALIAS);
        if (level == GLOBAL && idIsGlobal) return true;
        if (level == BOTH && idIsGlobal) return true;
        if (level == CI && idIsGlobal) return false;

        if (level != BOTH && level != CI) {
            return false;
        }

        if (root == null) {
            return true;
        }

        return id.startsWith(root.getRootNodeName());
    }
}
