package com.xebialabs.deployit.service.comparison;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import org.springframework.stereotype.Component;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

@Component
public class Comparator {

    private static final String PASSWORD_OBFUSCATION = "********";

    public ListMultimap<String, String> compare(ConfigurationItem reference, List<ConfigurationItem> entities) {
        ComputingLinkedMap comparison = new ComputingLinkedMap(entities.size() + 1);

        checkCiTypes(reference.getType(), entities);

        Descriptor descriptor = DescriptorRegistry.getDescriptor(reference.getType());
        int nrCi = 0;

        List<ConfigurationItem> entitiesToCompare = Lists.newArrayList();
        entitiesToCompare.add(reference);
        entitiesToCompare.addAll(entities);
        for (ConfigurationItem compareEntity : entitiesToCompare) {
          writeFields(comparison, compareEntity, nrCi);
          writeProperties(comparison, compareEntity, descriptor, nrCi);
          nrCi++;
        }

        LinkedListMultimap<String, String> retval = LinkedListMultimap.create();
        for (String k : comparison.keySet()) {
            retval.putAll(k, comparison.get(k));
        }
        return retval;
    }

    @SuppressWarnings("unchecked")
    private void writeProperties(final ComputingLinkedMap comparison, final ConfigurationItem ci, final Descriptor descriptor, final int nrCi) {
        final Collection<PropertyDescriptor> nonHiddenDescriptors = Collections2.filter(descriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
            public boolean apply(PropertyDescriptor input) {
                return !input.isHidden();
            }
        });
        for (PropertyDescriptor propertyDescriptor : nonHiddenDescriptors) {
            final String key = propertyDescriptor.getName();
            Object value = propertyDescriptor.get(ci);
            if (propertyDescriptor.isPassword()) {
                value = PASSWORD_OBFUSCATION;
            }
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                case CI:
                    comparison.put(key, value != null ? value.toString() : null, nrCi);
                    break;
                case LIST_OF_STRING:
                case LIST_OF_CI:
                case SET_OF_STRING:
                case SET_OF_CI:
                    handleCollectionOfString(comparison, key, (Collection<String>) value, nrCi);
                    break;
                case MAP_STRING_STRING:
                    handleMap(comparison, key, (Map<String, String>) value, nrCi);
                    break;
                default:
                    throw new IllegalStateException("Should not end up here!");
            }
        }
    }

    private void handleMap(ComputingLinkedMap comparison, String key, Map<String, String> value, int nrCi) {
        if (value != null) {
            // Order the keys in the Map
            SortedSet<String> set = new TreeSet<String>(value.keySet());
            for (String k : set) {
                comparison.put(key + ": " + k, value.get(k), nrCi);
            }
        }
    }

    private void handleCollectionOfString(final ComputingLinkedMap comparison, final String key, final Collection<String> value, final int nrCi) {
        if (value != null) {
            final ArrayList<String> list = new ArrayList<String>(value);
            Collections.sort(list);
            comparison.put(key, Joiner.on(",").join(list), nrCi);
        }
    }

    private void writeFields(final ComputingLinkedMap comparison, final ConfigurationItem entity, int nrCi) {
        comparison.put("id", entity.getId(), nrCi);
        comparison.put("type", entity.getType().toString(), nrCi);
    }

    private void checkCiTypes(final Type referenceType, final List<ConfigurationItem> entities) {
        for (ConfigurationItem entity : entities) {
            checkArgument(entity.getType().equals(referenceType), "Not all configuration items are of type %s", referenceType);
        }
    }

    @SuppressWarnings("serial")
    static class ComputingLinkedMap extends LinkedHashMap<String, List<String>> {
        private String[] strings;

        public ComputingLinkedMap(int listSize) {
            super();
            strings = new String[listSize];
            Arrays.fill(strings, "");
        }

        @Override
        public List<String> get(Object o) {
            List<String> value = super.get(o);
            if (value == null) {
                value = Lists.newArrayList(strings);
                super.put((String) o, value);
            }

            return value;
        }

        public void put(String key, String value, int position) {
            List<String> values = get(key);
            values.remove(position);
            values.add(position, value);
        }
    }
}
