package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

@Component
public class DeployedService {
    private DeployedGenerator deployedGenerator;
    private TypeCalculator typeCalculator;

    @Autowired
    public DeployedService(DeployedGenerator deployedGenerator, TypeCalculator typeCalculator) {
        this.deployedGenerator = deployedGenerator;
        this.typeCalculator = typeCalculator;
    }

    @SuppressWarnings("rawtypes")
    public ListMultimap<Boolean, ConfigurationItem> generateAllDeployeds(Version pkg, Environment env) {
        Set<Container> containers = env.getMembers();
        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(env.getDictionaries());
        List<Deployed> validDeployeds = generateDeployedsOfType(pkg.getDeployables(), containers, null, dictionary);
        if (validDeployeds.isEmpty()) {
            throw new Checks.IncorrectArgumentException("Could not generate any deployeds for " + pkg.getId() + " and " + env.getId());
        }
        return createValidDeployedMap(validDeployeds, Lists.<Deployed>newArrayList());
    }

    @SuppressWarnings("rawtypes")
    public ListMultimap<Boolean, ConfigurationItem> generateSelectedDeployeds(List<ConfigurationItem> deployableCis, Environment env) {
        Set<Container> containers = env.getMembers();
        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(env.getDictionaries());

        List<Deployable> deployables = Lists.transform(deployableCis, new Function<ConfigurationItem, Deployable>() {
            @Override
            public Deployable apply(ConfigurationItem from) {
                checkArgument(from instanceof Deployable, "The entity %s is not a deployable", from.getId());
                return (Deployable) from;
            }
        });

        return createValidDeployedMap(generateDeployedsOfType(deployables, containers, null, dictionary), Lists.<Deployed>newArrayList());
    }

    @SuppressWarnings("rawtypes")
    public ListMultimap<Boolean, ConfigurationItem> createSelectedDeployed(Deployable deployable, Container container, Type deployedType, Environment env) {
        logger.debug("Creating deployed for [{}] and [{}]", deployable, container);

        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(env.getDictionaries());
        return createValidDeployedMap(generateDeployedsOfType(newArrayList(deployable), newArrayList(container), deployedType, dictionary, true), Lists.<Deployed>newArrayList());
    }

    @SuppressWarnings("rawtypes")
    public ListMultimap<Boolean, ConfigurationItem> generateUpgradedDeployeds(Version newPackage, DeployedApplication deployment) {
        final Environment environment = deployment.getEnvironment();
        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(environment.getDictionaries());
        List<Deployed> upgradeDeployeds = newArrayList();
        List<Deployed> oldDeployeds = newArrayList();
        for (final Deployed deployed : deployment.getDeployeds()) {
            Collection<Deployable> filtered = Collections2.filter(newPackage.getDeployables(), new Predicate<Deployable>() {
                @Override
                public boolean apply(Deployable input) {
                    return isSimilar(input, deployed.getDeployable());
                }
            });

            if (filtered.size() == 0) {
                oldDeployeds.add(deployedGenerator.generateUpgradedDeployed(deployed.getDeployable(), deployed, dictionary));
            } else if (filtered.size() == 1) {
                upgradeDeployeds.add(deployedGenerator.generateUpgradedDeployed(filtered.iterator().next(), deployed, dictionary));
            } else {
                throw new IllegalArgumentException(format("More than 1 applicable deployable found in new package for %s, will not fly.", deployed));
            }
        }

        return createValidDeployedMap(upgradeDeployeds, oldDeployeds);
    }

    @SuppressWarnings("rawtypes")
    private ListMultimap<Boolean, ConfigurationItem> createValidDeployedMap(List<Deployed> validDeployeds, List<Deployed> invalidDeployeds) {
        ListMultimap<Boolean, ConfigurationItem> deployedsMap = ArrayListMultimap.create();
        deployedsMap.putAll(Boolean.TRUE, validDeployeds);
        deployedsMap.putAll(Boolean.FALSE, invalidDeployeds);

        return deployedsMap;
    }

    private boolean isSimilar(Deployable newDeployable, Deployable deployable) {
        return newDeployable.getType().equals(deployable.getType()) && newDeployable.getName().equals(deployable.getName());
    }

    @SuppressWarnings("rawtypes")
    private List<Deployed> generateDeployedsOfType(Collection<Deployable> deployables, Collection<Container> containers, Type deployedType, ConsolidatedDictionary dictionary) {
        return generateDeployedsOfType(deployables, containers, deployedType, dictionary, false);
    }

    @SuppressWarnings("rawtypes")
    private List<Deployed> generateDeployedsOfType(Collection<Deployable> deployables, Collection<Container> containers, Type deployedType, ConsolidatedDictionary dictionary, boolean manualAction) {
        List<Deployed> deployeds = newArrayList();
        for (Deployable deployable : deployables) {
            for (Container container : containers) {
                if (deployedType != null) {
                    if (typeCalculator.findDeployedTypesForDeployableAndContainerTypes(deployable.getType(), container.getType()).contains(deployedType)) {
                        deployeds.add(deployedGenerator.generateDeployed(deployable, container, deployedType, dictionary));
                    }
                } else {
                    Deployed deployed;
                    if (manualAction) {
                        logger.debug("Creating manual deployed for [{}] and [{}]", deployable, container);
                        deployed = deployedGenerator.createMostSpecificDeployed(deployable, container, dictionary);
                    } else {
                        logger.debug("Creating automatic deployed for [{}] and [{}]", deployable, container);
                        deployed = deployedGenerator.generateMostSpecificDeployed(deployable, container, dictionary);
                    }
                    if (deployed != null) {
                        deployeds.add(deployed);
                    }
                }
            }
        }
        return deployeds;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedService.class);
}
