package com.xebialabs.deployit.service.deployment;

import java.io.StringWriter;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.deployment.planner.Plan;
import com.xebialabs.deployit.deployment.planner.Planner;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.WorkdirCleanerTrigger;
import com.xebialabs.deployit.task.step.RepositoryUpdateStep;

import static com.xebialabs.deployit.task.TaskType.INITIAL;
import static com.xebialabs.deployit.task.TaskType.UNDEPLOY;
import static com.xebialabs.deployit.task.TaskType.UPGRADE;

@Component
public class DeploymentService {

    private final Planner planner;

    @Autowired
    public DeploymentService(Planner planner) {
        this.planner = planner;
    }

    @SuppressWarnings("rawtypes")
    public DeltaSpecification prepareInitialDeployment(final DeployedApplication deployedApp) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().initial(deployedApp);
        DeploymentOperationCalculator.calculate(builder, Sets.<Deployed>newHashSet(), deployedApp.getDeployeds());
        return builder.build();
    }

    public DeltaSpecification prepareUpgradeDeployment(final DeployedApplication newDeployment, final DeployedApplication existingDeployment) {
        final DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().upgrade(existingDeployment, newDeployment);

        logger.trace("Incoming Deployeds {}", newDeployment.getDeployeds());
        logger.trace("Existing Deployeds {}", existingDeployment.getDeployeds());

        DeploymentOperationCalculator.calculate(builder, existingDeployment.getDeployeds(), newDeployment.getDeployeds());

        return builder.build();
    }

    @SuppressWarnings("rawtypes")
    public DeltaSpecification prepareUndeployment(final DeployedApplication deployedApplication) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().undeploy(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, deployedApplication.getDeployeds(), Sets.<Deployed>newHashSet());
        return builder.build();
    }

    public TaskSpecification getTaskSpecification(DeltaSpecification deltaSpecification, WorkDir... workdirs) {
        TaskSpecification spec = null;
        switch (deltaSpecification.getOperation()) {
            case CREATE:
                spec = getTaskSpecification("Initial deployment of " + deltaSpecification.getDeployedApplication().getId(), deltaSpecification, workdirs);
                addMetadata(spec, deltaSpecification.getDeployedApplication(), INITIAL);
                break;
            case DESTROY:
                spec = getTaskSpecification("Undeployment of " + deltaSpecification.getPreviousDeployedApplication().getId(), deltaSpecification, workdirs);
                addMetadata(spec, deltaSpecification.getPreviousDeployedApplication(), UNDEPLOY);
                break;
            case MODIFY:
                String description = "Upgrade deployment of " + deltaSpecification.getPreviousDeployedApplication().getId();
                spec = getTaskSpecification(description, deltaSpecification, workdirs);
                addMetadata(spec, deltaSpecification.getDeployedApplication(), UPGRADE);
                break;
            case NOOP:
                break;
        }

        return spec;

    }

    private TaskSpecification getTaskSpecification(final String description, final DeltaSpecification specification, final WorkDir... workDirs) {
        unsetTokensForRealityPush(specification);
        Plan plan = planner.plan(specification);
        logger.info("Generated plan:\n{}", plan.writePlan(new StringWriter()));

        TaskSpecification spec = new TaskSpecification(description, Permissions.getAuthentication().getName(), getNonEmptyStepList(plan));
        spec.getListeners().addAll(plan.getListeners());
        spec.getListeners().add(new WorkdirCleanerTrigger(workDirs));
        spec.getListeners().add(new RepositoryUpdateTrigger(specification));
        spec.getListeners().add(new PartialCommitTrigger(specification, plan.getCheckpoints()));

        return spec;
    }

    private static void unsetTokensForRealityPush(final DeltaSpecification specification) {
        specification.getDeployedApplication().set$token(null);

        for (Delta delta : specification.getDeltas()) {
            if (delta.getDeployed() instanceof BaseConfigurationItem) {
                ((BaseConfigurationItem) delta.getDeployed()).set$token(null);
            }
        }
    }

    private static void addMetadata(TaskSpecification spec, DeployedApplication deployedApp, TaskType taskType) {
        spec.getMetadata().put("environment_id", deployedApp.getEnvironment().getId());
        spec.getMetadata().put("environment", deployedApp.getEnvironment().getName());
        spec.getMetadata().put("version", deployedApp.getVersion().getName());
        spec.getMetadata().put("application", deployedApp.getVersion().getApplication().getName());
        spec.getMetadata().put("taskType", taskType.name());
    }

    private static List<Step> getNonEmptyStepList(Plan plan) {
        if (plan.getSteps().isEmpty()) {
            return Lists.<Step>newArrayList(new RepositoryUpdateStep());
        }
        return plan.getSteps();
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);
}
