package com.xebialabs.deployit.service.deployment;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.deployment.planner.Plan;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;
import com.xebialabs.deployit.engine.tasker.TaskStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.repository.ChangeSet;
import com.xebialabs.deployit.repository.ConfigurationItemCloner;
import com.xebialabs.deployit.repository.RepositoryServiceHolder;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.CANCELLED;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.STOPPED;
import static com.xebialabs.deployit.engine.spi.execution.Transitions.checkTransition;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;

@SuppressWarnings("serial")
public class PartialCommitTrigger implements ExecutionStateListener {

    private DeltaSpecification specification;
    private DeployedApplication inprogressDeployedApplication;
    private List<Plan.Checkpoint> checkpoints;

    private List<Plan.Checkpoint> checkpointed = newArrayList();

    public PartialCommitTrigger(DeltaSpecification specification, List<Plan.Checkpoint> checkpoints) {
        this.specification = specification;
        if (specification.getOperation() == Operation.CREATE) {
            this.inprogressDeployedApplication = ConfigurationItemCloner.cloneShallow(specification.getDeployedApplication());
            this.inprogressDeployedApplication.getDeployeds().clear();
        } else {
            this.inprogressDeployedApplication = ConfigurationItemCloner.cloneShallow(specification.getPreviousDeployedApplication());
        }
        this.checkpoints = checkpoints;
    }

    @Override
    public void stepStateChanged(final StepExecutionStateEvent event) {
        final TaskStep step = (TaskStep) event.step();
        // SKIPPED or DONE steps are checkpointed.
        if (!step.getState().isFinal()) {
            return;
        }

        Iterable<Plan.Checkpoint> matchingCheckpoints = Iterables.filter(checkpoints, new Predicate<Plan.Checkpoint>() {
            public boolean apply(Plan.Checkpoint input) {
                return input.getStep().equals(step.getImplementation());
            }
        });

        for (Plan.Checkpoint checkpoint : matchingCheckpoints) {
            makeCheckpoint(checkpoint);
        }
    }

    private void makeCheckpoint(final Plan.Checkpoint checkpoint) {
        logger.debug("Checkpointing...");
        Optional<Plan.Checkpoint> optionalCheckpoint = Iterables.tryFind(checkpointed, new Predicate<Plan.Checkpoint>() {
            @Override
            public boolean apply(final Plan.Checkpoint input) {
                return input.getDelta().equals(checkpoint.getDelta());
            }
        });

        boolean added = false;
        if (optionalCheckpoint.isPresent()) {
            Plan.Checkpoint foundCheckpoint = optionalCheckpoint.get();
            checkpointed.remove(foundCheckpoint);
            if (foundCheckpoint.getDelta().getOperation() == MODIFY) {
                added = doModifyCheckpoint(foundCheckpoint, checkpoint);
            }
        }

        if (!added) {
            checkpointed.add(checkpoint);
        }
    }

    private boolean doModifyCheckpoint(final Plan.Checkpoint foundCheckpoint, final Plan.Checkpoint checkpoint) {
        if (foundCheckpoint.getOperation() == DESTROY && checkpoint.getOperation() == CREATE) {
            checkpointed.add(new Plan.Checkpoint(checkpoint.getDelta(), checkpoint.getStep()));
            return true;
        }
        return false;
    }

    @Override
    public void taskStateChanged(final TaskExecutionStateEvent event) {
        if (checkTransition(event, STOPPED, CANCELLED)) {
            doPartialCommit();
        }
    }

    private void doPartialCommit() {
        if (specification.getOperation() == CREATE && checkpointed.isEmpty()) {
            return;
        }

        ChangeSet cs = new ChangeSet();
        for (Plan.Checkpoint checkpoint : checkpointed) {
            Deployed<?,?> deployed = checkpoint.getDelta().getDeployed();
            Deployed<?,?> previousDeployed = checkpoint.getDelta().getPrevious();
            switch (checkpoint.getOperation()) {
            case CREATE:
                inprogressDeployedApplication.addDeployed(deployed);
                cs.createOrUpdate(deployed);
                break;
            case MODIFY:
            case NOOP:
                inprogressDeployedApplication.getDeployeds().remove(previousDeployed);
                inprogressDeployedApplication.addDeployed(deployed);
                cs.createOrUpdate(deployed);
                break;
            case DESTROY:
                inprogressDeployedApplication.getDeployeds().remove(previousDeployed);
                cs.delete(previousDeployed);
                break;
            }
        }

        if (inprogressDeployedApplication.getDeployeds().isEmpty()) {
            cs.delete(inprogressDeployedApplication);
        } else {
            cs.createOrUpdate(inprogressDeployedApplication);
        }


        RepositoryServiceHolder.getRepositoryService().execute(cs);
    }

    DeltaSpecification getSpecification() {
        return specification;
    }

    DeployedApplication getInprogressDeployedApplication() {
        return inprogressDeployedApplication;
    }

    private static final Logger logger = LoggerFactory.getLogger(PartialCommitTrigger.class);
}