package com.xebialabs.deployit.service.deployment;

import java.util.List;

import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.tasker.Engine;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.engine.tasker.TaskExecutionContext;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.WorkdirCleanerTrigger;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class RollbackService {

    private Engine engine;

    private DeploymentService deploymentService;

    @Autowired
    public RollbackService(Engine engine, final DeploymentService deploymentService) {
        this.engine = engine;
        this.deploymentService = deploymentService;
    }

    public TaskSpecification rollback(Task task) {
        TaskExecutionContext context = task.getContext();
        WorkdirCleanerTrigger workdirCleanerTrigger = getAndUnregisterWorkdirCleaner(context);
        PartialCommitTrigger partialCommitTrigger = getPartialCommitTrigger(context);
        archive(task);
        DeltaSpecification deltaSpecification = partialCommitTrigger.getSpecification();
        DeployedApplication newDeployedApplication = findNewDeployedApplication(task, deltaSpecification, partialCommitTrigger.getInprogressDeployedApplication());
        TaskSpecification rollbackSpecification = createRollbackSpecification(task, deltaSpecification, newDeployedApplication, workdirCleanerTrigger.getWorkDirs());
        rollbackSpecification.getMetadata().put("taskType", TaskType.ROLLBACK.name());
        rollbackSpecification.getMetadata().put("rollbackTask", task.getId());
        return rollbackSpecification;
    }

    private TaskSpecification createRollbackSpecification(final Task task, final DeltaSpecification deltaSpecification, final DeployedApplication newDeployedApplication, final List<WorkDir> workDirs) {
        DeployedApplication previousDeployedApplication = deltaSpecification.getPreviousDeployedApplication();
        DeltaSpecification newDeltaSpecification = null;
        switch (deltaSpecification.getOperation()) {
            case CREATE:
                newDeltaSpecification = deploymentService.prepareUndeployment(newDeployedApplication);
                break;
            case DESTROY:
                if (task.getState() == TaskExecutionState.DONE){
                    newDeltaSpecification = deploymentService.prepareInitialDeployment(previousDeployedApplication);
                } else {
                    newDeltaSpecification = deploymentService.prepareUpgradeDeployment(previousDeployedApplication, newDeployedApplication);
                }
                break;
            case MODIFY:
                newDeltaSpecification = deploymentService.prepareUpgradeDeployment(previousDeployedApplication, newDeployedApplication);
                break;
            case NOOP:
                throw new IllegalStateException("DeltaSpecification can never have a NOOP operation.");
        }
        return deploymentService.getTaskSpecification(newDeltaSpecification, workDirs.toArray(new WorkDir[workDirs.size()]));
    }

    private WorkdirCleanerTrigger getAndUnregisterWorkdirCleaner(TaskExecutionContext context) {
        String name = WorkdirCleanerTrigger.class.getName();
        WorkdirCleanerTrigger trigger = (WorkdirCleanerTrigger) context.getAttribute(name);
        context.unsetAttribute(name);
        return trigger;
    }

    private PartialCommitTrigger getPartialCommitTrigger(TaskExecutionContext context) {
        String name = PartialCommitTrigger.class.getName();
        return (PartialCommitTrigger) context.getAttribute(name);
    }

    private void archive(Task task) {
        if (task.getState() == TaskExecutionState.EXECUTED) {
            engine.archive(task.getId());
        } else if (task.getState() == TaskExecutionState.STOPPED) {
            engine.cancel(task.getId());
        } else {
            throw new IllegalStateException(String.format("Can only rollback a STOPPED or EXECUTED task [%s (%s)]", task.getId(), task.getState()));
        }
    }

    private DeployedApplication findNewDeployedApplication(final Task task, final DeltaSpecification deltaSpecification, final DeployedApplication inprogressDeployedApplication) {
        if (task.getState() == TaskExecutionState.CANCELLED) {
            return inprogressDeployedApplication;
        } else if (task.getState() == TaskExecutionState.DONE) {
            return deltaSpecification.getDeployedApplication();
        } else {
            throw new IllegalStateException(String.format("Task %s should be CANCELLED or DONE now, not [%s].", task.getId(), task.getState()));
        }
    }
}
