package com.xebialabs.deployit.service.importer;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportedPackage;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.importer.ListableImporter;
import com.xebialabs.deployit.server.api.importer.PackageInfo;
import com.xebialabs.deployit.service.importer.reader.DeployableConfigurationItemReader;
import com.xebialabs.deployit.service.importer.reader.DeployitJarManifest;
import com.xebialabs.deployit.service.importer.reader.ManifestEntry;
import com.xebialabs.deployit.util.GuavaFiles;

import de.schlichtherle.truezip.file.TFile;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.sort;

public class ManifestBasedDarImporter implements ListableImporter {
    public static final String TEMPORARY_FILES = "temporaryFiles";

    @Override
    public List<String> list(File directory) {
        List<String> packages = newArrayList(PackageScanner.scan(directory));
        sort(packages);
        return packages;
    }

    @Override
    public boolean canHandle(ImportSource source) {
        File file = source.getFile();
        return PackageScanner.isDarPackage(file) || PackageScanner.isExplodedPackage(file);
    }

    @Override
    public PackageInfo preparePackage(ImportSource source, ImportingContext ctx) {
        PackageInfo packageInfo = new PackageInfo(source);
        updatePackageInfoWithManifestValues(packageInfo);
        ctx.setAttribute(TEMPORARY_FILES, Lists.<TFile>newArrayList());
        return packageInfo;
    }

    private static void updatePackageInfoWithManifestValues(final PackageInfo packageInfo) {
        DeployitJarManifest manifest = getManifest(packageInfo);
        
        packageInfo.setApplicationName(manifest.getApplicationName());
        packageInfo.setApplicationVersion(manifest.getApplicationVersion());
    }

    @Override
    public ImportedPackage importEntities(PackageInfo packageInfo, ImportingContext ctx) {
        ImportedPackage importedPackage = new ImportedPackage(packageInfo);
        DeployitJarManifest manifest = getManifest(packageInfo);
        createDeployableEntities(packageInfo, manifest, importedPackage, ctx);
        return importedPackage;
    }

    private static void createDeployableEntities(PackageInfo packageInfo, DeployitJarManifest manifest, ImportedPackage importedPackage, ImportingContext ctx) {
        DeployableConfigurationItemReader reader = new DeployableConfigurationItemReader(manifest);
        for (ManifestEntry  entry : manifest.getEntries()) {
            if (reader.isMiddlewareResource(entry)) {
                importedPackage.addDeployable(reader.readMiddlewareConfiguration((DeploymentPackage) importedPackage.getVersion(), entry));
            } else {
                importedPackage.addDeployable(reader.readArtifact((DeploymentPackage) importedPackage.getVersion(), packageInfo.getSource(), entry, ctx));
            }
        }
    }

    private static DeployitJarManifest getManifest(PackageInfo packageInfo) {
        TFile manifestFileForUnpackedPackage = PackageScanner.getManifestFileFromImportSource(packageInfo.getSource());
        try {
            return DeployitJarManifest.readFromFile(manifestFileForUnpackedPackage);
        } finally {
            umountSource(packageInfo);
        }
    }

    private static void umountSource(PackageInfo packageInfo) {
        try {
            TFile.umount(new TFile(packageInfo.getSource().getFile()));
        } catch (Exception e) {
        }
    }

    @Override
    public void cleanUp(PackageInfo packageInfo, ImportingContext ctx) {
        List<TFile> files = ctx.getAttribute(TEMPORARY_FILES);
        for (TFile file : files) {
            logger.debug("Cleaning up temporary file {}", file);
            try {
                File javaFile = file.getFile();
                if (javaFile.isDirectory()) {
                    GuavaFiles.deleteRecursively(javaFile);
                } else {
                    if (!javaFile.delete()) {
                        logger.info("Couldn't delete file: {}", javaFile);
                    }
                }                
            } catch (IOException e) {
                logger.error("Couldn't clean up file {}", file, e);
            }
        }
        umountSource(packageInfo);
    }

    private static final Logger logger = LoggerFactory.getLogger(ManifestBasedDarImporter.class);
}
