package com.xebialabs.deployit.service.importer.reader;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.service.importer.ManifestBasedDarImporter;
import com.xebialabs.overthere.local.LocalFile;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.fs.FsSyncException;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static java.lang.String.format;

public class DeployableConfigurationItemReader {

    static final String NAMESPACE_SEPARATOR = "/";

    static final String CI = "CI-";

    static final String TYPE_ATTRIBUTE_NAME = "CI-Type";

    static final String NAME_ATTRIBUTE_NAME = "CI-Name";
    private final DeployitJarManifest manifest;

    public DeployableConfigurationItemReader(DeployitJarManifest manifest) {
        this.manifest = manifest;
    }

    public Deployable readMiddlewareConfiguration(DeploymentPackage deploymentPackage, ManifestEntry entry) {

        String configurationItemType = readCIType(entry);

        Descriptor descriptor = getDescriptorForConfigurationItem(entry.getName(), configurationItemType);

        Deployable configurationItem = descriptor.newInstance();
        final String baseId = deploymentPackage.getId();
        configurationItem.setId(baseId + "/" + getDeployableNameFromEntryName(entry.getName()));
        fillAttributes(configurationItem, descriptor, entry, baseId);
        return configurationItem;
    }

    public boolean isMiddlewareResource(ManifestEntry entry) {
        String configurationItemType = entry.getValue(TYPE_ATTRIBUTE_NAME);

        if (isBlank(configurationItemType)) {
            return false;
        }

        Descriptor configurationItemDescriptor = getDescriptorForConfigurationItem(entry.getName(), configurationItemType);

        // it is a deployable non-artifact
        checkArgument(
                configurationItemDescriptor.isAssignableTo(Deployable.class),
                "Configuration item %s of type %s is not Deployable", 
                entry.getName(),
                configurationItemType);
        
        return !configurationItemDescriptor.isAssignableTo(DeployableArtifact.class);
    }

    public DeployableArtifact readArtifact(DeploymentPackage deploymentPackage, ImportSource source, ManifestEntry entry, ImportingContext ctx) {
        String configurationItemType = readCIType(entry);
        Descriptor descriptor = getDescriptorForConfigurationItem(entry.getName(), configurationItemType);
        DeployableArtifact artifact = descriptor.newInstance();

        final String baseId = deploymentPackage.getId();
        artifact.setId(baseId + "/" + getDeployableName(entry));
        // Need to wrap in TFile to detect this archivedetector
        TFile sourceArchive = new TFile(source.getFile());
        try {
            TFile tempFolder = createTempFolderForImport(ctx, artifact);
            // Prevent going deep in archives by setting archivedetector to NULL.
            TFile dest = copyArtifactData(sourceArchive, entry.getName(), tempFolder, artifact, baseId);
            logger.debug("Adding {} {} to deployment package", descriptor.getType(), dest);
            artifact.setFile(LocalFile.valueOf(dest.getFile()));
            fillAttributes(artifact, descriptor, entry, baseId);
            return artifact;
        } finally {
            if (sourceArchive.isArchive()) {
                try {
                    TFile.umount(sourceArchive);
                } catch (FsSyncException e) {
                    logger.error("Unable to release resources for archive {}", sourceArchive.getName());
                    logger.error("Following exception occurred while trying to release resources: {}", e.getMessage());
                }
            }
        }
    }

    private static TFile copyArtifactData(TFile sourceArchive, String entryName, TFile tempFolder, DeployableArtifact artifact, String baseId) {
        TFile artifactFile = new TFile(sourceArchive, entryName, TArchiveDetector.NULL);
        if (!artifactFile.exists()) {
            throw new RuntimeIOException(format("The entry %s could not be found in the importable package %s.", entryName, baseId));
        }

        TFile dest = new TFile(tempFolder, artifactFile.getName());

        try {
            artifactFile.cp_r(dest);
        } catch (IOException e) {
            throw new RuntimeIOException(format("Could not copy %s to %s while importing %s", artifactFile, dest, artifact.getId()), e);
        }

        return dest;
    }

    private static TFile createTempFolderForImport(ImportingContext ctx, Artifact artifact) {
        try {
            String name = "temp-" + artifact.getName();
            TFile tempFolder = new TFile(File.createTempFile(name, "")).rm().mkdir(false);
            ctx.<List<TFile>>getAttribute(ManifestBasedDarImporter.TEMPORARY_FILES).add(tempFolder);
            logger.debug("Created Temporary folder {}", tempFolder);
            return tempFolder;
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private static String getDeployableName(ManifestEntry entry) {
        String name = entry.getValue(NAME_ATTRIBUTE_NAME);
        if (name != null) {
            return name;
        } else {
            return getDeployableNameFromEntryName(entry.getName());
        }
    }

    protected static String getDeployableNameFromEntryName(String entryName) {
        String name;
        int index = entryName.lastIndexOf(NAMESPACE_SEPARATOR);
        if (index > -1) {
            index++;
            if (index >= entryName.length()) {
                throw new ImporterException("Could not determine artifact Name: %s because entry ends with '/'", entryName);
            }
            name = entryName.substring(index);
        } else {
            name = entryName;
        }
        return name;
    }

    private static String readCIType(final ManifestEntry entry) {
        return readCIType(entry, entry.getName());
    }

    private static String readCIType(final ManifestEntry entry, String name) {
        String configurationItemTypeShortName = entry.getValue(TYPE_ATTRIBUTE_NAME);
        if (isBlank(configurationItemTypeShortName)) {
            throw new ImporterException("Could not import Name: %s because of missing %s entry in MANIFEST.MF", name, TYPE_ATTRIBUTE_NAME);
        }
        return configurationItemTypeShortName;
    }

    private static Descriptor getDescriptorForConfigurationItem(String entryName, String configurationItemTypeShortName) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(configurationItemTypeShortName);
        if (descriptor == null) {
            throw new ImporterException("Could not import Name: %s because %s is not a known CI type", entryName, configurationItemTypeShortName);
        }
        return descriptor;
    }

    private static boolean isBlank(String packageFormatVersion) {
        return Strings.nullToEmpty(packageFormatVersion).trim().isEmpty();
    }

    void fillAttributes(final Deployable entity, final Descriptor configurationItemDescriptor,
                        final ManifestEntry entry, final String baseId) {
        for (PropertyDescriptor propertyDescriptor : configurationItemDescriptor.getPropertyDescriptors()) {
            final String name = propertyDescriptor.getName();
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                    final String value = entry.getValue(CI + name);
                    propertyDescriptor.set(entity, value);
                    break;
                case CI:
                    final String mfName = entry.getValue(CI + name);
                    if (mfName == null) {
                        continue;
                    }
                    Deployable configurationItem = convertReference(baseId, mfName);
                    propertyDescriptor.set(entity, configurationItem);
                    break;
                case SET_OF_STRING:
                    Set<String> stringsSet = newHashSet(entry.getStrings(propertyDescriptor.getName()));
                    if (!stringsSet.isEmpty()) {
                        propertyDescriptor.set(entity, stringsSet);
                    }
                    break;
                case SET_OF_CI:
                    Set<Deployable> setOfCis = newHashSet(transformCollectionOfCis(entry, baseId, propertyDescriptor));
                    if (!setOfCis.isEmpty()) {
                        propertyDescriptor.set(entity, setOfCis);
                    }
                    break;
                case LIST_OF_STRING:
                    List<String> stringsList = newArrayList(entry.getStrings(propertyDescriptor.getName()));
                    if (!stringsList.isEmpty()) {
                        propertyDescriptor.set(entity, stringsList);
                    }
                    break;
                case LIST_OF_CI:
                    List<Deployable> listOfCis = newArrayList(transformCollectionOfCis(entry, baseId, propertyDescriptor));
                    if (!listOfCis.isEmpty()) {
                        propertyDescriptor.set(entity, listOfCis);
                    }
                    break;
                case MAP_STRING_STRING:
                    Map<String, String> map = entry.getMap(propertyDescriptor.getName());
                    propertyDescriptor.set(entity, map);
                    break;
                default:
                    throw new IllegalStateException("Should not end up here!");

            }
        }
    }

    private Deployable convertReference(String baseId, String mfName) {
        ManifestEntry referencedAttrs = checkNotNull(manifest.getEntry(mfName), "Couldn't find the referenced entry [%s], are you using the entry Name?", mfName);
        Type ciType = Type.valueOf(readCIType(referencedAttrs, mfName));
        Deployable configurationItem = ciType.getDescriptor().newInstance();
        String ciName = referencedAttrs.getValue(NAME_ATTRIBUTE_NAME);
        if (ciName == null) {
            ciName = mfName;
        }
        String id = IdGenerator.generateId(baseId, ciName);
        configurationItem.setId(id);
        return configurationItem;
    }

    private Collection<Deployable> transformCollectionOfCis(final ManifestEntry entry, final String baseId, PropertyDescriptor propertyDescriptor) {
        return Collections2.transform(entry.getStrings(propertyDescriptor.getName()), new Function<String, Deployable>() {
            @Override
            public Deployable apply(String from) {
                return convertReference(baseId, from);
            }
        });
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployableConfigurationItemReader.class);

}
