package com.xebialabs.deployit.service.importer.reader;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.service.importer.ImporterException;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileInputStream;

public class DeployitJarManifest {

    private static final String PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME = "Deployit-Package-Format-Version";
    private static final String PACKAGE_FORMAT_VERSION_NUMBER = "1.3";
    private static final String APPLICATION_ATTRIBUTE_NAME = "CI-Application";
    private static final String VERSION_ATTRIBUTE_NAME = "CI-Version";

    private final Manifest manifest;

    private DeployitJarManifest(Manifest manifest) {
        this.manifest = manifest;

        verifyPackageFormatVersion();
    }

    public static DeployitJarManifest readFromFile(TFile manifestFile) {
        Manifest manifest;
        InputStream in = null;
        try {
            in = new TFileInputStream(manifestFile);
            manifest = new Manifest(in);
        } catch (IOException exception) {
            throw new RuntimeIOException("Cannot read manifest file " + manifestFile, exception);
        } finally {
            Closeables.closeQuietly(in);
        }

        return new DeployitJarManifest(manifest);
    }

    public String getApplicationName() {
        return getMainEntryAttribute(APPLICATION_ATTRIBUTE_NAME);
    }

    public String getApplicationVersion() {
        return getMainEntryAttribute(VERSION_ATTRIBUTE_NAME);
    }

    private String getMainEntryAttribute(final String attributeName) {
        String attributeValue = manifest.getMainAttributes().getValue(attributeName);
        if (attributeValue == null || attributeValue.trim().isEmpty()) {
            throw new ImporterException("Deployment package does not declare the " + attributeName + " main attribute");
        }
        return attributeValue;
    }

    private void verifyPackageFormatVersion() {
        String packageFormatVersion = getMainEntryAttribute(PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME);
        if (!PACKAGE_FORMAT_VERSION_NUMBER.equals(packageFormatVersion)) {
            throw new ImporterException("Deployment package does not have the right " + PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME + " header value (actual: "
                    + packageFormatVersion + " expected: " + PACKAGE_FORMAT_VERSION_NUMBER + ")");
        }
    }

    public Collection<ManifestEntry> getEntries() {
        // TODO do more efficiently
        List<ManifestEntry> entries = new ArrayList<ManifestEntry>();
        for (Map.Entry<String, Attributes> entry : manifest.getEntries().entrySet()) {
            entries.add(new ManifestEntry(entry.getKey(), entry.getValue()));
        }
        
        return entries;
    }

    public ManifestEntry getEntry(String key) {
        return new ManifestEntry(key, manifest.getAttributes(key));
    }
}
