package com.xebialabs.deployit.service.importer.source;

import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.util.GuavaFiles;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

public class UrlSource implements ImportSource {

    private final URL location;
    private FileSource downloaded;
    private final File tempDir;

    public UrlSource(URL location) {
        this.location = location;
        tempDir = Files.createTempDir();
    }

    private void download() {
        try {
            logger.debug("Preparing to download package from {}", location);
            HttpClient client = new HttpClient();
            URI uri = location.toURI();
            HttpMethod method = new GetMethod(uri.toString());
            int statusCode = client.executeMethod(method);
            if (statusCode != HttpStatus.SC_OK) {
                throw new RuntimeIOException("Failed to download package,status="+ statusCode +", from url " + location);
            }
           save(uri, method.getResponseBodyAsStream());
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        } catch (HttpException e) {
            throw new RuntimeIOException(e);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private void save(URI uri, InputStream responseBodyAsStream) throws IOException {
        try {
            String path = uri.getPath();
            String name = new File(path).getName();
            File archive = new File(tempDir, name);
            ByteStreams.copy(responseBodyAsStream, Files.newOutputStreamSupplier(archive));
            this.downloaded = new FileSource(archive, true);
        } finally {
            Closeables.closeQuietly(responseBodyAsStream);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(UrlSource.class);

    @Override
    public File getFile() {
        if (downloaded == null) {
            download();
        }

        logger.debug("Successfully downloaded file {}", downloaded.getFile().getName());
        return downloaded.getFile();
    }

    @Override
    public void cleanUp() {
        if (downloaded != null) {
            downloaded.cleanUp();
        }

        GuavaFiles.deleteQuietly(tempDir);
    }

    @Override
    public String toString() {
        return "UrlSource[" + location + "]";
    }
}
