package com.xebialabs.deployit.service.version.exporter;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.jar.Manifest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Maps;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.overthere.local.LocalFile;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileOutputStream;
import de.schlichtherle.truezip.fs.FsSyncException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

@Component
public class ExporterService {

    private final RepositoryService repositoryService;

    private final ManifestWriter manifestWriter;

    @Autowired
    public ExporterService(RepositoryService repositoryService, ManifestWriter manifestWriter) {
        this.repositoryService = repositoryService;
        this.manifestWriter = manifestWriter;
    }

    public LocalFile exportDar(String versionId, final WorkDir workDir) {
        checkNotNull(versionId, "id of the version to be exported is null");

        ConfigurationItem ci = repositoryService.read(versionId, workDir);
        checkArgument(ci.getType().instanceOf(Type.valueOf(DeploymentPackage.class)), "Expected udm.DeploymentPackage for exporting but got %s", ci.getType());
        DeploymentPackage deploymentPackage = (DeploymentPackage) ci;

        String name = String.format("%s-%s.dar", deploymentPackage.getApplication().getName(), deploymentPackage.getVersion());
        TFile exportedDar = new TFile(workDir.getPath(), name);
        try {
            exportedDar.mkdirs();

            addManifestToDar(deploymentPackage, exportedDar);

            copyArtifactsToDar(deploymentPackage, exportedDar);

            return (LocalFile) LocalFile.valueOf(exportedDar.getFile());
        } catch (Exception e) {
            throw new RuntimeException("Errors found while exporting package version", e);
        } finally {
            try {
                TFile.umount(exportedDar);
            } catch (FsSyncException ignore) {
            }
        }
    }

    private void addManifestToDar(DeploymentPackage deploymentPackage, TFile exportedDar) {
        TFileOutputStream out = null;
        try {
            TFile manifestFile = new TFile(exportedDar, "/META-INF/MANIFEST.MF");
            out = new TFileOutputStream(manifestFile);
            Manifest manifest = createManifest(deploymentPackage);
            manifest.write(out);
        } catch (Exception e) {
            throw new RuntimeException("Errors found while adding manifest file to dar", e);
        } finally {
            Closeables.closeQuietly(out);
        }
    }

    private Manifest createManifest(DeploymentPackage deploymentPackage) {
        manifestWriter.addHeader(deploymentPackage.getApplication().getName(), deploymentPackage.getVersion());
        Set<Deployable> deployables = deploymentPackage.getDeployables();
        for (Deployable each : deployables) {
            Map<String, String> values = getPropertyValues(each);
            if (isMiddlewareResource(each)) {
                manifestWriter.addResourceEntry(each.getName(), each.getType().toString(), values);
            } else {
                BaseDeployableArtifact artifact = (BaseDeployableArtifact) each;
                values.put("Name", each.getName());
                manifestWriter.addArtifactEntry(artifact.getFile().getName(), each.getType().toString(), values);
            }
        }
        Manifest manifest = manifestWriter.getManifest();
        return manifest;
    }

    private void copyArtifactsToDar(DeploymentPackage deploymentPackage, TFile exportedDar) throws IOException {
        // copy the artifacts to the exported DAR
        Set<Deployable> deployables = deploymentPackage.getDeployables();
        for (Deployable each : deployables) {
            if (!isMiddlewareResource(each)) {
                BaseDeployableArtifact artifact = (BaseDeployableArtifact) each;
                File srcArtifact = new File(artifact.getFile().getPath());
                TFile destArtifact = new TFile(exportedDar, artifact.getFile().getName(), TArchiveDetector.NULL);
                TFile.cp_r(srcArtifact, destArtifact, TArchiveDetector.NULL);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private Map<String, String> getPropertyValues(Deployable ci) {
        Map<String, String> valueMap = Maps.newHashMap();
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {

            if (propertyDescriptor.isHidden()) {
                continue;
            }

            Object value = propertyDescriptor.get(ci);
            if (value == null)
                continue;
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                    valueMap.put(propertyDescriptor.getName(), String.valueOf(value));
                    break;
                case CI:
                    valueMap.put(propertyDescriptor.getName(), ((BaseConfigurationItem) value).getName());
                    break;
                case SET_OF_STRING:
                case LIST_OF_STRING:
                    int strCount = 1;
                    for (String s : (Collection<String>) value) {
                        valueMap.put(propertyDescriptor.getName() + "-EntryValue-" + strCount++, s);
                    }
                    break;
                case SET_OF_CI:
                case LIST_OF_CI:
                    int ciCount = 1;
                    for (ConfigurationItem ciEntry : (Collection<ConfigurationItem>) value) {
                        valueMap.put(propertyDescriptor.getName() + "-EntryValue-" + ciCount++, ciEntry.getName());
                    }
                    break;
                case MAP_STRING_STRING:
                    for (Map.Entry<String, String> entry : ((Map<String, String>) value).entrySet()) {
                        valueMap.put(propertyDescriptor.getName() + "-" + entry.getKey(), entry.getValue());
                    }
                    break;
            }
        }
        return valueMap;
    }

    private boolean isMiddlewareResource(ConfigurationItem ci) {
        return !ci.getType().getDescriptor().isAssignableTo(DeployableArtifact.class);
    }

    @SuppressWarnings("unused")
    private static Logger logger = LoggerFactory.getLogger(ExporterService.class);

}
