package com.xebialabs.deployit.task;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.joda.time.DateTime;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

public class ArchivedTaskSearchParameters {


    public enum DateRange {
        AFTER, BEFORE, BETWEEN, NONE;
    }

    public enum Status {
        COMPLETED, CANCELLED, COMPLETED_AFTER_RETRY, ANY;
    }

    protected DateTime startDate;
    protected DateTime endDate;
    protected String executedBy;
    protected Set<Application> applications = newHashSet();
    protected Set<String> environments = newHashSet();
    protected String taskUuid;
    protected DateRange dateRangeSearch = ArchivedTaskSearchParameters.DateRange.NONE;
    protected Status status = ArchivedTaskSearchParameters.Status.ANY;
    protected EnumSet<TaskType> taskTypes = EnumSet.noneOf(TaskType.class);
    protected String orderBy;

    public ArchivedTaskSearchParameters() {
    }

    public ArchivedTaskSearchParameters(ArchivedTaskSearchParameters p) {
        this.startDate = p.startDate;
        this.endDate = p.endDate;
        this.executedBy = p.executedBy;
        this.applications = newHashSet(p.applications);
        this.environments = newHashSet(p.environments);
        this.dateRangeSearch = p.dateRangeSearch;
        this.status = p.status;
        this.taskUuid = p.taskUuid;
        this.taskTypes = p.taskTypes;
        this.orderBy = p.orderBy;
    }

    public ArchivedTaskSearchParameters orderBy(String property) {
        this.orderBy = property;
        return this;
    }

    public ArchivedTaskSearchParameters executedByUser(String user) {
        this.executedBy = user;
        return this;
    }

    public ArchivedTaskSearchParameters createdBetween(DateTime startDate, DateTime endDate) {
        checkNotNull(startDate, "Start date cannot be null");
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BETWEEN;
        if (startDate.isAfter(endDate)) {
            this.startDate = endDate;
            this.endDate = startDate;
        } else {
            this.startDate = startDate;
            this.endDate = endDate;
        }
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrBefore(DateTime endDate) {
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BEFORE;
        this.endDate = endDate;
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrAfter(DateTime startDate) {
        checkNotNull(startDate, "Start date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.AFTER;
        this.startDate = startDate;
        return this;
    }

    public ArchivedTaskSearchParameters forApplicationAndVersion(String application, String version) {
        Checks.checkTrue(environments.isEmpty(), "Can't query for both environments and applications.");

        applications.add(new Application(application, version));

        return this;
    }

    public ArchivedTaskSearchParameters forApplication(String application) {
        return forApplicationAndVersion(application, null);
    }

    public ArchivedTaskSearchParameters thatAreOfType(EnumSet<TaskType> types) {
        this.taskTypes = types;
        return this;
    }

    public ArchivedTaskSearchParameters forEnvironment(String environment) {
        Checks.checkTrue(applications.isEmpty(), "Can't query for both environments and applications.");

        environments.add(environment);

        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, List<ConfigurationItemId> cis) {
        for (ConfigurationItemId appOrEnvName : cis) {
            forFilterType(filterType, appOrEnvName);
        }

        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, ConfigurationItemId appOrEnvName) {
        switch (filterType) {
        case APPLICATION:
            forApplication(appOrEnvName.getId());
            break;
        case ENVIRONMENT:
            forEnvironment(appOrEnvName.getId());
            break;
        default:
            break;
        }

        return this;
    }

    public ArchivedTaskSearchParameters thatWasCancelled() {
        status = Status.CANCELLED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompleted() {
        status = Status.COMPLETED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompletedAfterRetry() {
        status = Status.COMPLETED_AFTER_RETRY;
        return this;
    }

    public ArchivedTaskSearchParameters withUniqueId(String uuid) {
        this.taskUuid = uuid;
        return this;
    }

    protected static class Application {
        public final String name;
        public final String version;

        Application(String name, String version) {
            this.name = name;
            this.version = version;
        }
    }
}
