package com.xebialabs.deployit.task.archive;

import java.util.List;

import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.InvalidQueryException;
import javax.jcr.query.QueryManager;
import javax.jcr.query.qom.*;

import com.xebialabs.deployit.task.TaskType;
import org.joda.time.DateTime;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;

import static com.xebialabs.deployit.jcr.JcrConstants.TASK_ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;
import static com.xebialabs.deployit.task.archive.JcrTaskArchive.*;
import static javax.jcr.query.qom.QueryObjectModelConstants.*;

public class JcrArchivedTaskSearchQueryBuilder extends ArchivedTaskSearchParameters {

    private final ValueFactory values;
    private final QueryObjectModelFactory queries;

    static final String TASK_SELECTOR_NAME = "task";

    public JcrArchivedTaskSearchQueryBuilder(QueryManager qm, ValueFactory vf, ArchivedTaskSearchParameters p) {
        super(p);

        this.values = vf;
        this.queries = qm.getQOMFactory();
    }

    public QueryObjectModel buildQuery() throws RepositoryException {
        List<Constraint> constraints = Lists.newArrayList();

        appendTaskUuidCriteria(constraints);
        appendTaskTypeCriteria(constraints);
        appendEnvironmentCriteria(constraints);
        appendApplicationCriteria(constraints);
        appendExecutedByCriteria(constraints);
        appendStatusCriteria(constraints);
        appendDateRangeCriteria(constraints);

        Ordering[] ordering = createOrderBy();

        Selector selector = queries.selector(TASK_NODETYPE_NAME, TASK_SELECTOR_NAME);
        return queries.createQuery(selector, and(constraints), ordering, null);
    }

    private Ordering[] createOrderBy() throws RepositoryException {
        if(this.orderBy != null ){
            Ordering[] ordering = new Ordering[] {
                queries.ascending(queries.propertyValue(TASK_SELECTOR_NAME, this.orderBy))};
            return ordering;
        } else {
            return null;
        }
    }

    private void appendTaskUuidCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(taskUuid))
            constraints.add(equalConstraint(TASK_ID_PROPERTY_NAME,taskUuid));
    }

    private static boolean isNotBlank(String s) {
        return !Strings.nullToEmpty(s).trim().isEmpty();
    }

    private void appendTaskTypeCriteria(List<Constraint> constraints) throws RepositoryException {
        if (taskTypes.isEmpty()) {
            return;
        }

        List<Constraint> taskTypeConstraints = Lists.newArrayList();
        for (TaskType tt : taskTypes) {
            taskTypeConstraints.add(taskTypeConstraint(tt));
        }
        constraints.add(or(taskTypeConstraints));
    }

    private void appendEnvironmentCriteria(List<Constraint> constraints) throws RepositoryException {
        if (environments.isEmpty()) {
            constraints.add(queries.descendantNode(TASK_SELECTOR_NAME, JcrConstants.TASKS_NODE_ID));
            return;
        }

        List<Constraint> enviromentConstraints = Lists.newArrayList();
        for (String environment : environments) {
            enviromentConstraints.add(environmentConstraint(environment));
        }
        constraints.add(or(enviromentConstraints));
    }

    private void appendApplicationCriteria(List<Constraint> constraints) throws RepositoryException {
        if (applications.isEmpty()) {
            return;
        }

        List<Constraint> applicationConstraints = Lists.newArrayList();
        for (Application application : applications) {
            applicationConstraints.add(applicationConstraint(application));
        }
        constraints.add(or(applicationConstraints));
    }

    private void appendExecutedByCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(executedBy)) {
            constraints.add(equalConstraint(OWNING_USER, executedBy));
        }
    }

    private void appendStatusCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (status) {
            case COMPLETED:
                constraints.add(queries.and(equalConstraint(STATE, TaskExecutionState.DONE.name()), equalConstraint(FAILURE_COUNT, 0l)));
                break;
            case CANCELLED:
                constraints.add(equalConstraint(STATE, TaskExecutionState.CANCELLED.name()));
                break;
            case COMPLETED_AFTER_RETRY:
                Constraint stateConstraint = equalConstraint(STATE, TaskExecutionState.DONE.name());
                Constraint failureConstraint = greaterThanConstraint(FAILURE_COUNT, 0l);
                constraints.add(queries.and(stateConstraint, failureConstraint));
                break;
        }
    }

    private void appendDateRangeCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (dateRangeSearch) {
            case AFTER:
                DateTime startDateAsCal = cloneAndSetTimeToStartOfDay(startDate);
                constraints.add(greaterThanOrEqualConstraint(START_DATE, startDateAsCal));
                break;

            case BEFORE:
                DateTime endDateAsCal = cloneAndSetTimeToEndOfDay(endDate);
                constraints.add(lessThanOrEqualConstraint(START_DATE, endDateAsCal));
                break;

            case BETWEEN:
                DateTime betweenStartDate = cloneAndSetTimeToStartOfDay(startDate);
                DateTime betweenEndDate = cloneAndSetTimeToEndOfDay(endDate);
                Constraint startConstraint = greaterThanOrEqualConstraint(START_DATE, betweenStartDate);
                Constraint endConstraint = lessThanOrEqualConstraint(START_DATE, betweenEndDate);
                constraints.add(startConstraint);
                constraints.add(endConstraint);
        }
    }

    private Constraint notEqualConstraint(String propName, String value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_NOT_EQUAL_TO, values.createValue(value));
    }

    private Constraint equalConstraint(String propName, String value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, values.createValue(value));
    }

    private Constraint equalConstraint(String propName, long value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, values.createValue(value));
    }

    private Constraint greaterThanConstraint(String propName, long value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_GREATER_THAN, values.createValue(value));
    }

    private Constraint greaterThanOrEqualConstraint(String propName, DateTime value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_GREATER_THAN_OR_EQUAL_TO, values.createValue(value.toGregorianCalendar()));
    }

    private Constraint lessThanOrEqualConstraint(String propName, DateTime value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_LESS_THAN_OR_EQUAL_TO, values.createValue(value.toGregorianCalendar()));
    }

    private Constraint addConstraint(String propName, String op, Value v) throws RepositoryException {
        return queries.comparison(queries.propertyValue(TASK_SELECTOR_NAME, propName), op, queries.literal(v));
    }

    private Constraint environmentConstraint(String environment) throws InvalidQueryException, RepositoryException {
        String path = JcrConstants.TASKS_NODE_ID + "/" + JcrTaskArchive.encodeEnvironmentId(environment);

        return queries.descendantNode(TASK_SELECTOR_NAME, path);
    }

    private Constraint applicationConstraint(Application application) throws RepositoryException {
        Constraint c = equalConstraint(APPLICATION, application.name);
        if (application.version != null) {
            c = queries.and(c, equalConstraint(VERSION, application.version));
        }
        return c;
    }

    private Constraint taskTypeConstraint(TaskType taskType) throws RepositoryException {
        Constraint c = equalConstraint(DEPLOYMENT_TYPE, taskType.toString());
        return c;
    }

    private Constraint and(List<Constraint> constraints) throws RepositoryException {
        Checks.checkTrue(!constraints.isEmpty(), "List of constraints may not be empty to combine into AND constraint.");

        Constraint andConstraint = constraints.get(0);
        for (int i = 1; i < constraints.size(); i++) {
            andConstraint = queries.and(andConstraint, constraints.get(i));
        }
        return andConstraint;
    }

    private Constraint or(List<Constraint> constraints) throws RepositoryException {
        Checks.checkTrue(!constraints.isEmpty(), "List of constraints may not be empty to combine into OR constraint.");

        Constraint orConstraint = constraints.get(0);
        for (int i = 1; i < constraints.size(); i++) {
            orConstraint = queries.or(orConstraint, constraints.get(i));
        }
        return orConstraint;
    }

    private static DateTime cloneAndSetTimeToStartOfDay(DateTime date) {
        return date.withHourOfDay(0).withMinuteOfHour(0).withSecondOfMinute(0).withMillisOfSecond(0);
    }

    private static DateTime cloneAndSetTimeToEndOfDay(DateTime date) {
        return date.withHourOfDay(23).withMinuteOfHour(59).withSecondOfMinute(59).withMillisOfSecond(999);
    }
}
