package com.xebialabs.deployit.task.archive;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;
import com.xebialabs.deployit.engine.tasker.Archive;
import com.xebialabs.deployit.engine.tasker.TaskNotFoundException;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.grouping.GroupBy;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.TaskType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.jcr.*;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.query.qom.QueryObjectModel;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.jcr.JcrConstants.*;
import static com.xebialabs.deployit.jcr.JcrUtils.*;
import static com.xebialabs.deployit.task.archive.JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME;

@Component
public class JcrTaskArchive implements Archive {

    public static final String ENVIRONMENTS_ROOT = "Environments/";

    static final String STATE = "state";
    static final String START_DATE = "startDate";
    static final String COMPLETION_DATE = "completionDate";
    static final String LAST_MODIFICATION_DATE = "lastModificationDate";
    static final String NR_OF_STEPS = "nrOfSteps";
    static final String CURRENT_STEP_NR = "currentStepNr";
    static final String OWNING_USER = "ownerUser";
    static final String FAILURE_COUNT = "failureCount";
    static final String LOG = "log";
    static final String DESCRIPTION = "description";
    static final String APPLICATION = "application";
    static final String VERSION = "version";
    static final String ENVIRONMENT_NAME = "environment";
    static final String ENVIRONMENT_ID = "environment_id";
    static final String DEPLOYMENT_TYPE = "deploymentType";

    private final JcrTemplate jcrTemplate;

    private static final Pattern qualifiedTaskIdPattern = Pattern.compile("^(.*)/(.*)/(.*)/(.*)$"); // ie. env/app/version/uuid
    private final Set<String> EXCLUDE_SPECIAL_ENV_TASK_NAMES = newHashSet(ENVIRONMENTS_ROOT + TaskType.INSPECTION,
            ENVIRONMENTS_ROOT + TaskType.CONTROL, ENVIRONMENTS_ROOT + "rep:policy");

    @Autowired
    public JcrTaskArchive(final JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public void archive(final TaskWithSteps task) {
        if (isDeploymentTask(task)) {
            saveDeploymentTask(task);
        } else {
            saveTask(task);
        }
    }

    private void saveTask(final TaskWithSteps task) {
        jcrTemplate.execute(new JcrCallback<String>() {
            @Override
            public String doInJcr(Session session) throws RepositoryException {
                Node jcrTaskNode = createTaskNode(task, getTasksRootNode(session));
                fillTaskNode(jcrTaskNode, task);
                session.save();
                return null;
            }
        });
    }

    private void saveDeploymentTask(final TaskWithSteps task) {
        checkDeploymentTask(task);
        jcrTemplate.execute(new JcrCallback<String>() {
            @Override
            public String doInJcr(Session session) throws RepositoryException {
                Node jcrTaskNode = createDeploymentTaskNode(task, getTasksRootNode(session));
                fillDeploymentTaskNode(jcrTaskNode, task);
                session.save();
                return null;
            }
        });
    }

    private static Node createTaskNode(TaskWithSteps task, Node tasksRootNode) throws RepositoryException {
        Node taskType = getOrCreateChild(tasksRootNode, task.getMetadata().get("taskType"));
        Node node = taskType.addNode(task.getId());
        node.addMixin(TASK_NODETYPE_NAME);
        return node;
    }

    private static Node createDeploymentTaskNode(final TaskWithSteps task, Node tasksRootNode) throws RepositoryException {
        Node envNode = getOrCreateChild(tasksRootNode, encodeEnvironmentId(task.getMetadata().get("environment_id")));
        Node appNode = getOrCreateChild(envNode, task.getMetadata().get("application"));
        Node verNode = getOrCreateChild(appNode, task.getMetadata().get("version"));
        Node node = verNode.addNode(task.getId());
        node.addMixin(TASK_NODETYPE_NAME);
        return node;
    }

    private static void checkDeploymentTask(TaskWithSteps task) {
        checkState(!Strings.nullToEmpty(task.getMetadata().get("application")).isEmpty(), "applicationName in deployment task must be set");
        checkState(!Strings.nullToEmpty(task.getMetadata().get("version")).isEmpty(), "applicationVersion in deployment task must be set");
        checkState(!Strings.nullToEmpty(task.getMetadata().get("environment")).isEmpty(), "environment in deployment task must be set");
        checkState(!Strings.nullToEmpty(task.getMetadata().get("environment_id")).isEmpty(), "environment_id in deployment task must be set");
    }

    static String encodeEnvironmentId(String id) {
        if (id.startsWith(ENVIRONMENTS_ROOT)) {
            id = id.substring(ENVIRONMENTS_ROOT.length());
        }
        return id.replace("/", "$#");
    }

    static String decodeEnvironmentId(String id) {
        id = id.replace("$#", "/");
        if (!id.startsWith(ENVIRONMENTS_ROOT)) {
            return ENVIRONMENTS_ROOT + id;
        }
        return id;
    }

    private static void fillDeploymentTaskNode(Node taskNode, final TaskWithSteps task) throws RepositoryException {
        fillTaskNode(taskNode, task);
        taskNode.setProperty(APPLICATION, task.getMetadata().get("application"));
        taskNode.setProperty(VERSION, task.getMetadata().get("version"));
        taskNode.setProperty(ENVIRONMENT_NAME, task.getMetadata().get("environment"));
        taskNode.setProperty(ENVIRONMENT_ID, encodeEnvironmentId(task.getMetadata().get("environment_id")));
    }

    private static void fillTaskNode(Node taskNode, TaskWithSteps task) throws RepositoryException {
        taskNode.setProperty(TASK_ID_PROPERTY_NAME, task.getId());
        taskNode.setProperty(STATE, task.getState().name());
        taskNode.setProperty(START_DATE, task.getStartDate().toGregorianCalendar());
        taskNode.setProperty(COMPLETION_DATE, task.getCompletionDate().toGregorianCalendar());
        taskNode.setProperty(CURRENT_STEP_NR, task.getCurrentStepNr());
        taskNode.setProperty(NR_OF_STEPS, task.getNrSteps());
        taskNode.setProperty(OWNING_USER, task.getOwner());
        taskNode.setProperty(FAILURE_COUNT, task.getFailureCount());
        taskNode.setProperty(DEPLOYMENT_TYPE, task.getMetadata().get("taskType"));

        int nr = 1;
        for (StepState step : task.getSteps()) {
            Node stepNode = taskNode.addNode(STEP_NODE_NAME_PREFIX + nr++);
            stepNode.setProperty(DESCRIPTION, step.getDescription());
            stepNode.setProperty(STATE, step.getState().name());
            stepNode.setProperty(START_DATE, step.getStartDate() == null ? null : step.getStartDate().toGregorianCalendar());
            stepNode.setProperty(COMPLETION_DATE, step.getCompletionDate() == null ? null : step.getCompletionDate().toGregorianCalendar());
            stepNode.setProperty(FAILURE_COUNT, step.getFailureCount());
            stepNode.setProperty(LOG, step.getLog());
        }
    }

    private static boolean isDeploymentTask(TaskWithSteps task) {
        return EnumSet.of(TaskType.ROLLBACK, TaskType.INITIAL, TaskType.UPGRADE, TaskType.UNDEPLOY).contains(TaskType.valueOf(task.getMetadata().get("taskType")));
    }

    public TaskWithSteps getTask(final String taskId) {
        Matcher matcher = qualifiedTaskIdPattern.matcher(taskId);
        if (matcher.matches()) {
            String internalId = encodeEnvironmentId(matcher.group(1)) + "/" + matcher.group(2) + "/" + matcher.group(3) + "/" + matcher.group(4);
            return getTaskUsingFullyQualifiedPath(internalId);
        } else {
            return getTaskUsingUuid(taskId);
        }

    }

    private TaskWithSteps getTaskUsingFullyQualifiedPath(final String fullyQualifiedPath) {
        return jcrTemplate.execute(new JcrCallback<ArchivedTask>() {
            @Override
            public ArchivedTask doInJcr(Session session) throws RepositoryException {
                try {
                    Node tasksNode = getTasksRootNode(session);
                    Node taskNode = tasksNode.getNode(fullyQualifiedPath);
                    ArchivedTask task = toTaskWithoutSteps(taskNode);
                    if (task == null) {
                        throw new NotFoundException("Cannot load task " + fullyQualifiedPath + " because that object with that id is not a task");
                    }
                    task.setSteps(getSteps(taskNode));
                    return task;
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException("Cannot load task " + fullyQualifiedPath + " because it does not exist");

                }
            }
        });
    }

    private TaskWithSteps getTaskUsingUuid(String taskUuid) {
        ArchivedTaskSearchParameters taskSearch = new ArchivedTaskSearchParameters();
        taskSearch.withUniqueId(taskUuid);
        Collection<ArchivedTask> deploymentTaskInfos = searchTasks(taskSearch);
        if (deploymentTaskInfos.size() == 1) {
            return deploymentTaskInfos.iterator().next();
        } else if (deploymentTaskInfos.isEmpty()) {
            throw new TaskNotFoundException("archive", taskUuid);
        } else {
            throw new NotFoundException("Cannot load task " + taskUuid + " because there are multiple tasks with same id. " + deploymentTaskInfos);
        }
    }

    private static ArchivedTask toTaskWithoutSteps(Node taskNode) throws RepositoryException {
        if (!taskNode.isNodeType(TASK_NODETYPE_NAME)) {
            return null;
        }

        ArchivedTask task = new ArchivedTask();
        task.setId(getProperty(taskNode, TASK_ID_PROPERTY_NAME));
        task.getMetadata().put("taskType", getProperty(taskNode, DEPLOYMENT_TYPE));
        task.setState(TaskExecutionState.valueOf(getProperty(taskNode, STATE)));
        task.setStartDate(getDate(taskNode, START_DATE));
        task.setCompletionDate(getDate(taskNode, COMPLETION_DATE));
        task.setOwner(getOptionalProperty(taskNode, OWNING_USER));
        task.setCurrentStepNr(getIntegerProperty(taskNode, CURRENT_STEP_NR));
        task.setFailureCount(getIntegerProperty(taskNode, FAILURE_COUNT));

        if (isDeploymentTask(task)) {
            task.getMetadata().put("environment", getProperty(taskNode, ENVIRONMENT_NAME));
            if (taskNode.hasProperty(ENVIRONMENT_ID)) {
                task.getMetadata().put("environment_id", decodeEnvironmentId(getProperty(taskNode, ENVIRONMENT_ID)));
            } else {
                // Support old nodes that do not have the environment_id set.
                task.getMetadata().put("environment_id", decodeEnvironmentId(getProperty(taskNode, ENVIRONMENT_NAME)));
            }
            task.getMetadata().put("application", getProperty(taskNode, APPLICATION));
            task.getMetadata().put("version", getProperty(taskNode, VERSION));
        }

        return task;
    }


    private static List<StepState> getSteps(final Node taskNode) throws RepositoryException {
        int nrOfSteps = getIntegerProperty(taskNode, NR_OF_STEPS);
        List<StepState> steps = Lists.newArrayList();
        for (int nr = 1; nr <= nrOfSteps; nr++) {
            Node stepNode = taskNode.getNode(STEP_NODE_NAME_PREFIX + nr);
            steps.add(toStep(stepNode));
        }
        return steps;
    }

    private static ArchivedStep toStep(Node stepNode) throws RepositoryException {
        ArchivedStep step = new ArchivedStep();
        step.setDescription(getProperty(stepNode, DESCRIPTION));
        step.setState(StepExecutionState.valueOf(getProperty(stepNode, STATE)));
        step.setStartDate(getDate(stepNode, START_DATE));
        step.setCompletionDate(getDate(stepNode, COMPLETION_DATE));
        step.setLog(getProperty(stepNode, LOG));
        step.setFailureCount(getIntegerProperty(stepNode, FAILURE_COUNT));
        return step;
    }

    private static Node getTasksRootNode(Session session) throws RepositoryException {
        return session.getRootNode().getNode(TASKS_NODE_NAME);
    }

    public Collection<ArchivedTask> searchTasks(final ArchivedTaskSearchParameters params) {
        return searchTasks(params, true);
    }

    public Collection<ArchivedTask> searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params) {
        return searchTasks(params, false);
    }

    private List<ArchivedTask> searchTasks(final ArchivedTaskSearchParameters params, final boolean loadSteps) {
        return jcrTemplate.execute(new JcrCallback<List<ArchivedTask>>() {
            @Override
            public List<ArchivedTask> doInJcr(Session session) throws RepositoryException {
                QueryResult queryResult = executeJcrQuery(session, params);
                final List<ArchivedTask> tasks = Lists.newArrayList();
                mapQueryResultToTasks(queryResult, loadSteps, new TaskCallback() {
                    @Override
                    public void doWithTask(ArchivedTask t) {
                        tasks.add(t);
                    }
                });
                return tasks;
            }
        });
    }

    public void searchTasks(final ArchivedTaskSearchParameters params, final TaskCallback callback) {
        searchTasks(params, callback, true);
    }

    public void searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params, final TaskCallback callback) {
        searchTasks(params, callback, false);
    }

    private Object searchTasks(final ArchivedTaskSearchParameters params, final TaskCallback callback, final boolean loadSteps) {
        return jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                QueryResult queryResult = executeJcrQuery(session, params);
                mapQueryResultToTasks(queryResult, loadSteps, callback);
                return null;
            }
        });
    }

    private static QueryResult executeJcrQuery(Session session, ArchivedTaskSearchParameters params) throws RepositoryException {
        QueryManager qm = session.getWorkspace().getQueryManager();
        JcrArchivedTaskSearchQueryBuilder queryBuilder = new JcrArchivedTaskSearchQueryBuilder(qm, session.getValueFactory(), params);
        QueryObjectModel query = queryBuilder.buildQuery();
        return query.execute();
    }

    public List<String> getAllEnvironments() {

        return jcrTemplate.execute(new JcrCallback<List<String>>() {
            @Override
            public List<String> doInJcr(Session session) throws RepositoryException {
                List<String> envs = newArrayList();
                NodeIterator nodes = session.getNode(TASKS_NODE_ID).getNodes();
                while (nodes.hasNext()) {
                    String envId = decodeEnvironmentId(nodes.nextNode().getName());
                    if (!EXCLUDE_SPECIAL_ENV_TASK_NAMES.contains(envId)) {
                        envs.add(envId);
                    }
                }
                return envs;
            }
        });
    }

    private static void mapQueryResultToTasks(QueryResult result, boolean loadSteps, TaskCallback callback) throws RepositoryException {
        final RowIterator iterator = result.getRows();
        while (iterator.hasNext()) {
            try {
                Node taskNode = iterator.nextRow().getNode(TASK_SELECTOR_NAME);
                ArchivedTask task = toTaskWithoutSteps(taskNode);
                if (loadSteps) {
                    task.setSteps(getSteps(taskNode));
                }
                callback.doWithTask(task);
            } catch (RepositoryException rre) {
                // Ignore, we weren't allowed to read a node, or one of
                // it's relations.
            }
        }
    }

    public List<String> getAllTaskIds() {
        return jcrTemplate.execute(new JcrCallback<List<String>>() {
            @Override
            public List<String> doInJcr(Session session) throws RepositoryException {
                QueryResult queryResult = executeJcrQuery(session, new ArchivedTaskSearchParameters());
                return getTaskIds(queryResult);
            }
        });
    }

    private static List<String> getTaskIds(final QueryResult queryResult) throws RepositoryException {
        List<String> taskIds = Lists.newArrayList();
        final RowIterator iterator = queryResult.getRows();
        while (iterator.hasNext()) {
            try {
                taskIds.add(getProperty(iterator.nextRow().getNode(JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME), TASK_ID_PROPERTY_NAME));
            } catch (RepositoryException rre) {
                // Ignore, we weren't allowed to read a node, or one of
                // it's relations.
            }
        }
        return taskIds;
    }

    @SuppressWarnings("unchecked")
    public Collection<Map<String, Object>> searchTasksWithoutLoadingSteps(final ArchivedTaskSearchParameters params, final GroupBy groupBy) {
        return (Collection<Map<String, Object>>) jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Collection<Map<String, Object>> doInJcr(Session session) throws RepositoryException {
                RowIterator iterator = executeJcrQuery(session, params).getRows();
                while (iterator.hasNext()) {
                    Node taskNode = iterator.nextRow().getNode(JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME);
                    groupBy.process(toTaskWithoutSteps(taskNode));
                }
                return groupBy.getResult();
            }
        });
    }

    public interface TaskCallback {
        void doWithTask(ArchivedTask t);
    }

}
