package com.xebialabs.deployit.upgrade;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Multimaps.index;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_NAME;
import static com.xebialabs.deployit.server.api.upgrade.Version.VERSION_0;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import nl.javadude.scannit.Scannit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimaps;
import com.xebialabs.deployit.booter.local.PluginVersions;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;

@Component
public class Upgrader {

    private static final String DEPLOYIT_COMPONENT = "deployit";

    private ApplicationContext context;

    private JcrTemplate jcrTemplate;
    
    private ListMultimap<String, Upgrade> upgradeBeans;

    private boolean questionAsked = false;
    
    public @Autowired Upgrader(ApplicationContext context, JcrTemplate jcrTemplate) {
        this.context = context;
        this.jcrTemplate = jcrTemplate;
    }

    @SuppressWarnings("unchecked")
    @PostConstruct
    public void doUpgrade() {
        upgradeBeans = getUpgrades();
        Set<String> registeredPlugins = PluginVersions.getRegisteredPlugins();
        upgradeComponent(DEPLOYIT_COMPONENT);
        for (String component : Iterables.concat(registeredPlugins)) {
            if (DEPLOYIT_COMPONENT.equals(component)) {
                continue;
            }
            upgradeComponent(component);
        }
    }


    void upgradeComponent(String component) {
        logger.debug("Checking component [{}] for upgrades", component);

        final Version componentVersion = readVersionOfComponent(component);
        logger.debug("Component [{}] has version [{}] in the Deployit repository", component, componentVersion);
        List<Upgrade> applicableUpgrades = filterUpgrades(ensureNotNull(upgradeBeans.get(component)), componentVersion);
        if (!applicableUpgrades.isEmpty()) {
            Collections.sort(applicableUpgrades);
            logger.info("Checked component [{}] which is at version [{}] -> Found upgrades to run: {}", new Object[] {component, componentVersion, applicableUpgrades});
            if (!questionAsked) {
                System.out.println("*** WARNING ***");
                System.out.println("We detected that we need to upgrade your Deployit repository");
                System.out.println("Before continuing we suggest you backup your Deployit repository in case the upgrade fails.");
                System.out.println("Please ensure you have 'INFO' level logging configured.");
                System.out.print("Please enter 'yes' if you want to continue [no]: ");
                String response = read();
                if (!"yes".equalsIgnoreCase(response)) {
                    logger.error("Did not receive an affirmative response on running upgrades, shutting down.");
                    EventBusHolder.publish(new ShutdownEvent());
                    return;
                }
                questionAsked = true;
            }
            applyUpgrades(applicableUpgrades);
        } else if (componentVersion.equals(Version.valueOf(component, VERSION_0))) {
            Version currentPluginVersion = Version.valueOf(component, PluginVersions.getVersionFor(component));
            logger.info("Registering previous unregistered version: [{}]", currentPluginVersion);
            storeVersionOfComponent(currentPluginVersion);
        }
    }

    void storeVersionOfComponent(final Version version) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                Node node = getVersionsNode(session);
                node.setProperty(version.getComponent(), version.getVersion());
                session.save();
                return null;
            }
        });
    }

    Version readVersionOfComponent(final String component) {
        return jcrTemplate.execute(new JcrCallback<Version>() {
            @Override
            public Version doInJcr(Session session) throws RepositoryException {
                Node node = getVersionsNode(session);
                if (node.hasProperty(component)) {
                    Property versionProp = node.getProperty(component);
                    return Version.valueOf(component, versionProp.getString());
                } else {
                    return Version.valueOf(component, VERSION_0);
                }
            }
        });
    }

    private Node getVersionsNode(Session session) throws RepositoryException {
        try {
            return session.getNode(VERSIONS_NODE_ID);
        } catch (PathNotFoundException pnfe) {
            return session.getRootNode().addNode(VERSIONS_NODE_NAME);
        }
    }

    private void applyUpgrades(final List<Upgrade> applicableUpgrades) {
        final ImmutableListMultimap<Version,Upgrade> versionToUpgradeMap = Multimaps.index(applicableUpgrades, new Function<Upgrade, Version>() {
            public Version apply(Upgrade input) {
                return input.upgradeVersion();
            }
        });

        List<Version> versions = newArrayList(Lists.transform(applicableUpgrades, new Function<Upgrade, Version>() {
            public Version apply(Upgrade input) {
                return input.upgradeVersion();
            }
        }));

        Collections.sort(versions);

        for (final Version version : versions) {
            logger.info("Upgrading to version [{}]", version);
            jcrTemplate.execute(new JcrCallback<Object>() {
                @Override
                public Object doInJcr(Session session) throws RepositoryException {
                    for (Upgrade applicableUpgrade : versionToUpgradeMap.get(version)) {
                        if (!applicableUpgrade.doUpgrade(new RawRepositoryImpl(session))) {
                            throw new UpgradeException("Could not perform upgrade %s to upgrade to %s", applicableUpgrade.getClass(), applicableUpgrade.upgradeVersion());
                        }
                    }

                    // Do a per-component transaction.
                    session.save();
                    return null;
                }
            });
            storeVersionOfComponent(version);
        }
    }

    private List<Upgrade> filterUpgrades(Collection<Upgrade> upgradeBeans, final Version repoVersion) {
        if (repoVersion.getVersion().equals(VERSION_0)) {
            return Lists.newArrayList();
        }
        List<Upgrade> applicableUpgrades = newArrayList(filter(upgradeBeans, new Predicate<Upgrade>() {
            @Override
            public boolean apply(Upgrade input) {
                return input.shouldBeApplied(repoVersion);
            }
        }));

        Collections.sort(applicableUpgrades);
        return applicableUpgrades;
    }

    protected ListMultimap<String, Upgrade> getUpgrades() {
        Set<Class<? extends Upgrade>> upgradeClasses = Scannit.getInstance().getSubTypesOf(Upgrade.class);
        logger.debug("Found the following upgraders: [{}]", upgradeClasses);
        return index(transform(upgradeClasses, new Function<Class<? extends Upgrade>, Upgrade>() {
            @Override
            public Upgrade apply(Class<? extends Upgrade> input) {
                return context.getAutowireCapableBeanFactory().createBean(input);
            }
        }), new Function<Upgrade, String>() {
            @Override
            public String apply(Upgrade input) {
                return input.upgradeVersion().getComponent();
            }
        });
    }

    protected String read() {
        BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
        try {
            final String line = stdin.readLine();
            if (line != null) {
                return line.trim();
            } else
                return line;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static <T> List<T> ensureNotNull(List<T> list) {
        return list == null ? Lists.<T>newArrayList() : list;
    }
    
    private static final Logger logger = LoggerFactory.getLogger(Upgrader.class);
}
